/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.LinearGradient;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;
import ej.microvg.VectorImageBuilder;

/**
 * Tests the vector image builder.
 */
@SuppressWarnings("nls")
public class TestImageBuilder {

	private static final int ROWS_COLS_COUNT = 19;

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests building and drawing an empty image (no path).
	 */
	@Test
	public static void testEmptyImage() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		// check that nothing has been drawn
		TestUtilities.checkArea("empty image", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100);
	}

	/**
	 * Tests building and drawing an image with an empty path.
	 */
	@Test
	public static void testEmptyPath() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);
		Path path = new Path();
		builder.addPath(path, 0xffff0000, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		// check that nothing has been drawn
		TestUtilities.checkArea("empty path", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100);
	}

	/**
	 * Tests the image size.
	 */
	@Test
	public static void testImageSize() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		// adding a 400x400 red square
		Path path = new Path();
		path.moveTo(0, 0);
		path.lineToRelative(400, 0);
		path.lineToRelative(0, 400);
		path.lineToRelative(-400, 0);
		path.close();
		builder.addPath(path, 0xffff0000, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		// check that the red square is the size of the image (100x100)
		TestUtilities.checkArea("square", Colors.RED, 100, 100, 100, 100);
		// check the outer area
		TestUtilities.checkPeripheralArea("outside", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100, 100, 0);
	}

	/**
	 * Test that building an image with a null width throws an exception
	 */
	public static void testNullWidth() {

		Class<?> c = null;
		try {
			new VectorImageBuilder(0, 100).build();
		} catch (Exception e) {
			c = e.getClass();
		}
		assertEquals(c, IllegalArgumentException.class);
	}

	/**
	 * Test that building an image with a null height throws an exception
	 */
	public static void testNullHeight() {

		Class<?> c = null;
		try {
			new VectorImageBuilder(100, 0).build();
		} catch (Exception e) {
			c = e.getClass();
		}
		assertEquals(c, IllegalArgumentException.class);
	}

	/**
	 * Test that building an image with a negative width throws an exception
	 */
	public static void testNegativeWidth() {

		Class<?> c = null;
		try {
			new VectorImageBuilder(-1, 100).build();
		} catch (Exception e) {
			c = e.getClass();
		}
		assertEquals(c, IllegalArgumentException.class);
	}

	/**
	 * Test that building an image with a negative height throws an exception
	 */
	public static void testNegativeHeight() {

		Class<?> c = null;
		try {
			new VectorImageBuilder(100, -1).build();
		} catch (Exception e) {
			c = e.getClass();
		}
		assertEquals(c, IllegalArgumentException.class);
	}

	/**
	 * Tests building and drawing an image with two opaque paths with color fill.
	 */
	@Test
	public static void testAddOpaquePath() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		// adding a 100x100 red square
		Path path1 = new Path();
		path1.moveTo(0, 0);
		path1.lineToRelative(100, 0);
		path1.lineToRelative(0, 100);
		path1.lineToRelative(-100, 0);
		path1.close();
		builder.addPath(path1, 0xffff0000, FillType.WINDING);

		// adding a 50x50 blue square inside the larger square
		Path path2 = new Path();
		path2.moveTo(25, 25);
		path2.lineToRelative(50, 0);
		path2.lineToRelative(0, 50);
		path2.lineToRelative(-50, 0);
		path2.close();
		builder.addPath(path2, 0xff0000ff, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		// check the small blue square
		TestUtilities.checkArea("smaller square", Colors.BLUE, 125, 125, 50, 50);
		// check the outer red square
		TestUtilities.checkPeripheralArea("bigger square", Colors.RED, 125, 125, 50, 50, 25, 0);
		// check the outer area
		TestUtilities.checkPeripheralArea("outside", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100, 50, 0);
	}

	/**
	 * Tests building and drawing an image with two paths with semi-opaque color fill.
	 */
	@Test
	public static void testAddPathSemiOpaque() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		// adding a 100x100 semi-opaque red square
		Path path1 = new Path();
		path1.moveTo(0, 0);
		path1.lineToRelative(100, 0);
		path1.lineToRelative(0, 100);
		path1.lineToRelative(-100, 0);
		path1.close();
		builder.addPath(path1, 0x88ff0000, FillType.WINDING);

		// adding a 50x50 semi-opaque blue square inside the larger square
		Path path2 = new Path();
		path2.moveTo(25, 25);
		path2.lineToRelative(50, 0);
		path2.lineToRelative(0, 50);
		path2.lineToRelative(-50, 0);
		path2.close();
		builder.addPath(path2, 0x880000ff, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		// check the small blue square
		TestUtilities.checkArea("smaller square", 0x380088, 125, 125, 50, 50);
		// check the outer red square
		TestUtilities.checkPeripheralArea("bigger semi-opaque square", 0x880000, 125, 125, 50, 50, 25, 0);
		// check the outer area
		TestUtilities.checkPeripheralArea("outside", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100, 50, 0);
	}

	/**
	 * Tests building and drawing an image with two paths with transparent color fill.
	 */
	@Test
	public static void testAddPathTransparent() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		// adding a 100x100 transparent red square
		Path path1 = new Path();
		path1.moveTo(0, 0);
		path1.lineToRelative(100, 0);
		path1.lineToRelative(0, 100);
		path1.lineToRelative(-100, 0);
		path1.close();
		builder.addPath(path1, 0x00ff0000, FillType.WINDING);

		// adding a 50x50 transparent blue square inside the larger square
		Path path2 = new Path();
		path2.moveTo(25, 25);
		path2.lineToRelative(50, 0);
		path2.lineToRelative(0, 50);
		path2.lineToRelative(-50, 0);
		path2.close();
		builder.addPath(path2, 0x000000ff, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		// check the small blue square
		TestUtilities.checkArea("smaller square", TestUtilities.BACKGROUND_COLOR, 125, 125, 50, 50);
		// check the outer red square
		TestUtilities.checkPeripheralArea("bigger semi-opaque square", TestUtilities.BACKGROUND_COLOR, 125, 125, 50, 50,
				25, 0);
		// check the outer area
		TestUtilities.checkPeripheralArea("outside", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100, 50, 0);
	}

	/**
	 * Tests building and drawing an image with a path with fill type even odd.
	 */
	@Test
	public static void testAddPathEvenOdd() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		Path path = new Path();
		path.moveTo(0, 35);
		path.lineTo(100, 35);
		path.lineTo(20, 100);
		path.lineTo(50, 0);
		path.lineTo(80, 100);
		path.close();
		builder.addPath(path, 0xffff0000, FillType.EVEN_ODD);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 50, 50);
		display.flush();

		TestUtilities.check("top", 100, 55, g, Colors.RED);
		TestUtilities.check("left", 55, 85, g, Colors.RED);
		TestUtilities.check("right", 145, 85, g, Colors.RED);
		TestUtilities.check("bottom left", 73, 145, g, Colors.RED);
		TestUtilities.check("bottom right", 127, 145, g, Colors.RED);
		TestUtilities.check("center", 100, 100, g, TestUtilities.BACKGROUND_COLOR);
	}

	/**
	 * Tests building and drawing an image with a path with fill type winding.
	 */
	@Test
	public static void testAddPathWinding() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		Path path = new Path();
		path.moveTo(0, 35);
		path.lineTo(100, 35);
		path.lineTo(20, 100);
		path.lineTo(50, 0);
		path.lineTo(80, 100);
		path.close();
		builder.addPath(path, 0xffff0000, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 50, 50);
		display.flush();

		TestUtilities.check("top", 100, 55, g, Colors.RED);
		TestUtilities.check("left", 55, 85, g, Colors.RED);
		TestUtilities.check("right", 145, 85, g, Colors.RED);
		TestUtilities.check("bottom left", 73, 145, g, Colors.RED);
		TestUtilities.check("bottom right", 127, 145, g, Colors.RED);
		TestUtilities.check("center", 100, 100, g, Colors.RED);
	}

	/**
	 * Tests building and drawing an image with a path with gradient.
	 */
	@Test
	public static void testAddGradientPath() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
		builder.addGradientPath(path, gradient, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		TestUtilities.check("left", 100, 100, g, Colors.RED);
		TestUtilities.check("center", 149, 100, g, Colors.YELLOW);
		TestUtilities.check("right", 199, 100, g, 0xffffff);
	}

	/**
	 * Tests building and drawing an image with a path with semi-opaque gradient.
	 */
	@Test
	public static void testAddSemiOpaqueGradientPath() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0x88ff0000, 0x88ffff00, 0x88ffffff });
		builder.addGradientPath(path, gradient, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		TestUtilities.check("left", 100, 100, g, 0x880000);
		TestUtilities.check("center", 149, 100, g, 0x808000);
		TestUtilities.check("right", 199, 100, g, 0x808080);
	}

	/**
	 * Tests building and drawing an image with a path with transparent gradient.
	 */
	@Test
	public static void testAddTransparentGradientPath() {

		VectorImageBuilder builder = new VectorImageBuilder(100, 100);

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0x00ff0000, 0x00ffff00, 0x00ffffff });
		builder.addGradientPath(path, gradient, FillType.WINDING);
		VectorImage image = builder.build();

		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 100, 100);
		display.flush();

		TestUtilities.checkArea("square", TestUtilities.BACKGROUND_COLOR, 100, 100, 100, 100);
	}

	/**
	 * Tests building and drawing an image with multiple paths.
	 */
	@Test
	public static void testMultiplePaths() {

		Display display = Display.getDisplay();

		VectorImageBuilder builder = new VectorImageBuilder(display.getWidth(), display.getHeight());

		int pathWidth = display.getWidth() / ROWS_COLS_COUNT;
		int pathHeight = display.getHeight() / ROWS_COLS_COUNT;

		int x = 0;
		int y = 0;
		int count = 0;
		for (int row = 0; row < ROWS_COLS_COUNT; row++) {
			for (int col = 0; col < ROWS_COLS_COUNT; col++) {
				int color = (count & 1) == 0 ? 0xffff0000 : 0xff0000ff;
				builder.addPath(createPath(x, y), color, FillType.WINDING);
				x += pathWidth;
				count++;
			}
			y += pathHeight;
			x = 0;
		}

		VectorImage image = builder.build();

		GraphicsContext g = display.getGraphicsContext();
		VectorGraphicsPainter.drawImage(g, image, 0, 0);
		display.flush();

		x = 0;
		y = 0;
		count = 0;
		for (int row = 0; row < ROWS_COLS_COUNT; row++) {
			for (int col = 0; col < ROWS_COLS_COUNT; col++) {
				int color = (count & 1) == 0 ? 0xffff0000 : 0xff0000ff;
				TestUtilities.checkArea("box (" + row + "," + col + ")", color, x, y, pathWidth, pathHeight);
				x += pathWidth;
				count++;
			}
			y += pathHeight;
			x = 0;
		}
	}

	private static Path createPath(int x, int y) {
		Path path = new Path();
		path.moveTo(x, y);
		path.lineToRelative(100, 0);
		path.lineToRelative(0, 100);
		path.lineToRelative(-100, 0);
		path.close();
		return path;
	}

}
