/*
 * Copyright (C) 2008 The Android Open Source Project
 * Copyright (C) 2015-2021 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package android.net;

import java.net.DatagramPacket;
import java.net.DatagramSocket;
import java.net.InetAddress;
import java.net.UnknownHostException;

import ej.bon.Util;
import ej.util.message.Level;
import ej.util.message.basic.BasicMessageLogger;

/**
 * Simple SNTP client class for retrieving network time.
 *
 * Sample usage:
 *
 * <pre>
 * SntpClient client = new SntpClient();
 * if (client.requestTime("time.foo.com")) {
 * 	long now = client.getNtpTime() + Util.platformTimeMillis() - client.getNtpTimeReference();
 * }
 * </pre>
 */
public class SntpClient {

	private static final String LOG_CATEGORY = "SntpClient";
	private static final int LOG_ID_RESPONSE_RECEIVED = 0;
	private static final int LOG_ID_UNKNOWN_HOST = 1;
	private static final int LOG_ID_REQUEST_TIME_FAILED = 2;
	private static final int LOG_ID_REQUEST_EXCEPTION = 3;

	private static final int REFERENCE_TIME_OFFSET = 16;
	private static final int ORIGINATE_TIME_OFFSET = 24;
	private static final int RECEIVE_TIME_OFFSET = 32;
	private static final int TRANSMIT_TIME_OFFSET = 40;
	private static final int NTP_PACKET_SIZE = 48;
	private static final int NTP_PORT = 123;
	private static final int NTP_MODE_CLIENT = 3;
	private static final int NTP_MODE_SERVER = 4;
	private static final int NTP_MODE_BROADCAST = 5;
	private static final int NTP_VERSION = 3;
	private static final int NTP_LEAP_NOSYNC = 3;
	private static final int NTP_STRATUM_DEATH = 0;
	private static final int NTP_STRATUM_MAX = 15;
	// Number of seconds between Jan 1, 1900 and Jan 1, 1970
	// 70 years plus 17 leap days
	private static final long OFFSET_1900_TO_1970 = ((365L * 70L) + 17L) * 24L * 60L * 60L;
	// system time computed from NTP server response
	private long mNtpTime;
	// value of Util.platformTimeMillis() corresponding to mNtpTime
	private long mNtpTimeReference;
	// round trip time in milliseconds
	private long mRoundTripTime;

	@SuppressWarnings("serial")
	private static class InvalidServerReplyException extends Exception {
		public InvalidServerReplyException(String message) {
			super(message);
		}
	}

	/**
	 * Sends an SNTP request to the given host and processes the response.
	 *
	 * @param host
	 *            host name of the server.
	 * @param timeout
	 *            network timeout in milliseconds. the timeout doesn't include the DNS lookup time, and it applies to
	 *            each individual query to the resolved addresses of the NTP server.
	 * @return true if the transaction was successful.
	 */
	public boolean requestTime(String host, int timeout) {
		return requestTime(host, NTP_PORT, timeout);
	}

	/**
	 * Sends an SNTP request to the given host and processes the response.
	 *
	 * @param host
	 *            host name of the server.
	 * @param port
	 *            port of the server.
	 * @param timeout
	 *            network timeout in milliseconds.
	 * @return true if the transaction was successful.
	 */
	public boolean requestTime(String host, int port, int timeout) {
		try {
			final InetAddress[] addresses = InetAddress.getAllByName(host);
			for (int i = 0; i < addresses.length; i++) {
				final InetAddress addr = addresses[i];
				if (addr != null && requestTime(addr, port, timeout)) {
					return true;
				}
			}
		} catch (UnknownHostException e) {
			BasicMessageLogger.INSTANCE.log(Level.WARNING, LOG_CATEGORY, LOG_ID_UNKNOWN_HOST, host); // Unknown host:
																										// <host>
		}
		BasicMessageLogger.INSTANCE.log(Level.FINE, LOG_CATEGORY, LOG_ID_REQUEST_TIME_FAILED); // request time failed
		return false;
	}

	/**
	 * Sends an SNTP request to the given host and processes the response.
	 *
	 * @param address
	 *            the address of the server.
	 * @param port
	 *            the port of the server.
	 * @param timeout
	 *            network timeout in milliseconds. the timeout doesn't include the DNS lookup time, and it applies to
	 *            each individual query to the resolved addresses of the NTP server.
	 * @return true if the transaction was successful.
	 */
	/* default */ boolean requestTime(InetAddress address, int port, int timeout) {
		try (DatagramSocket socket = new DatagramSocket()) {
			socket.setSoTimeout(timeout);
			byte[] buffer = new byte[NTP_PACKET_SIZE];
			DatagramPacket request = new DatagramPacket(buffer, buffer.length, address, port);
			// set mode = 3 (client) and version = 3
			// mode is in low 3 bits of first byte
			// version is in bits 3-5 of first byte
			buffer[0] = NTP_MODE_CLIENT | (NTP_VERSION << 3);
			// get current time and write it to the request packet
			final long requestTime = System.currentTimeMillis();
			final long requestTicks = Util.platformTimeMillis();
			writeTimeStamp(buffer, TRANSMIT_TIME_OFFSET, requestTime);
			socket.send(request);
			// read the response
			DatagramPacket response = new DatagramPacket(buffer, buffer.length);
			socket.receive(response);
			final long responseTicks = Util.platformTimeMillis();
			final long responseTime = requestTime + (responseTicks - requestTicks);
			// extract the results
			final byte leap = (byte) ((buffer[0] >> 6) & 0x3);
			final byte mode = (byte) (buffer[0] & 0x7);
			final int stratum = buffer[1] & 0xff;
			final long originateTime = readTimeStamp(buffer, ORIGINATE_TIME_OFFSET);
			final long receiveTime = readTimeStamp(buffer, RECEIVE_TIME_OFFSET);
			final long transmitTime = readTimeStamp(buffer, TRANSMIT_TIME_OFFSET);
			final long referenceTime = readTimeStamp(buffer, REFERENCE_TIME_OFFSET);
			/* Do validation according to RFC */
			// TODO: validate originateTime == requestTime.
			checkValidServerReply(leap, mode, stratum, transmitTime, referenceTime);
			long roundTripTime = responseTicks - requestTicks - (transmitTime - receiveTime);
			// receiveTime = originateTime + transit + skew
			// responseTime = transmitTime + transit - skew
			// clockOffset = ((receiveTime - originateTime) + (transmitTime - responseTime))/2
			// = ((originateTime + transit + skew - originateTime) +
			// (transmitTime - (transmitTime + transit - skew)))/2
			// = ((transit + skew) + (transmitTime - transmitTime - transit + skew))/2
			// = (transit + skew - transit + skew)/2
			// = (2 * skew)/2 = skew
			long clockOffset = ((receiveTime - originateTime) + (transmitTime - responseTime)) / 2;
			BasicMessageLogger.INSTANCE.log(Level.FINE, LOG_CATEGORY, LOG_ID_RESPONSE_RECEIVED,
					Long.valueOf(roundTripTime), Long.valueOf(clockOffset)); // round trip: <roundTripTime> ms, clock
																				// offset: <clockOffset> ms
			// save our results - use the times on this side of the network latency
			// (response rather than request time)
			this.mNtpTime = responseTime + clockOffset;
			this.mNtpTimeReference = responseTicks;
			this.mRoundTripTime = roundTripTime;
		} catch (Exception e) {
			BasicMessageLogger.INSTANCE.log(Level.FINE, LOG_CATEGORY, LOG_ID_REQUEST_EXCEPTION, e); // request time
																									// failed: <e>
			return false;
		}
		return true;
	}

	/**
	 * Returns the time computed from the NTP transaction.
	 *
	 * @return time value computed from NTP server response.
	 */
	public long getNtpTime() {
		return this.mNtpTime;
	}

	/**
	 * Returns the reference clock value (value of Util.platformTimeMillis()) corresponding to the NTP time.
	 *
	 * @return reference clock corresponding to the NTP time.
	 */
	public long getNtpTimeReference() {
		return this.mNtpTimeReference;
	}

	/**
	 * Returns the round trip time of the NTP transaction
	 *
	 * @return round trip time in milliseconds.
	 */
	public long getRoundTripTime() {
		return this.mRoundTripTime;
	}

	private static void checkValidServerReply(byte leap, byte mode, int stratum, long transmitTime, long referenceTime)
			throws InvalidServerReplyException {
		if (leap == NTP_LEAP_NOSYNC) {
			throw new InvalidServerReplyException("unsynchronized server");
		}
		if ((mode != NTP_MODE_SERVER) && (mode != NTP_MODE_BROADCAST)) {
			throw new InvalidServerReplyException("untrusted mode: " + mode);
		}
		if ((stratum == NTP_STRATUM_DEATH) || (stratum > NTP_STRATUM_MAX)) {
			throw new InvalidServerReplyException("untrusted stratum: " + stratum);
		}
		if (transmitTime == 0) {
			throw new InvalidServerReplyException("zero transmitTime");
		}
		if (referenceTime == 0) {
			throw new InvalidServerReplyException("zero reference timestamp");
		}
	}

	/**
	 * Reads an unsigned 32 bit big endian number from the given offset in the buffer.
	 */
	private long read32(byte[] buffer, int offset) {
		byte b0 = buffer[offset];
		byte b1 = buffer[offset + 1];
		byte b2 = buffer[offset + 2];
		byte b3 = buffer[offset + 3];
		// convert signed bytes to unsigned values
		int i0 = ((b0 & 0x80) == 0x80 ? (b0 & 0x7F) + 0x80 : b0);
		int i1 = ((b1 & 0x80) == 0x80 ? (b1 & 0x7F) + 0x80 : b1);
		int i2 = ((b2 & 0x80) == 0x80 ? (b2 & 0x7F) + 0x80 : b2);
		int i3 = ((b3 & 0x80) == 0x80 ? (b3 & 0x7F) + 0x80 : b3);
		return ((long) i0 << 24) + ((long) i1 << 16) + ((long) i2 << 8) + i3;
	}

	/**
	 * Reads the NTP time stamp at the given offset in the buffer and returns it as a system time (milliseconds since
	 * January 1, 1970).
	 */
	private long readTimeStamp(byte[] buffer, int offset) {
		long seconds = read32(buffer, offset);
		long fraction = read32(buffer, offset + 4);
		// Special case: zero means zero.
		if (seconds == 0 && fraction == 0) {
			return 0;
		}
		return ((seconds - OFFSET_1900_TO_1970) * 1000) + ((fraction * 1000L) / 0x100000000L);
	}

	/**
	 * Writes system time (milliseconds since January 1, 1970) as an NTP time stamp at the given offset in the buffer.
	 */
	private void writeTimeStamp(byte[] buffer, int offset, long time) {
		// Special case: zero means zero.
		if (time == 0) {
			for (int i = 0; i < 8; i++) {
				buffer[offset + i] = (byte) 0x00;
			}
			return;
		}
		long seconds = time / 1000L;
		long milliseconds = time - seconds * 1000L;
		seconds += OFFSET_1900_TO_1970;
		// write seconds in big endian format
		buffer[offset++] = (byte) (seconds >> 24);
		buffer[offset++] = (byte) (seconds >> 16);
		buffer[offset++] = (byte) (seconds >> 8);
		buffer[offset++] = (byte) (seconds >> 0);
		long fraction = milliseconds * 0x100000000L / 1000L;
		// write fraction in big endian format
		buffer[offset++] = (byte) (fraction >> 24);
		buffer[offset++] = (byte) (fraction >> 16);
		buffer[offset++] = (byte) (fraction >> 8);
		// low order bits should be random data
		buffer[offset++] = (byte) (Math.random() * 255.0);
	}
}
