/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;

import ej.basictool.ArrayTools;
import ej.hoka.http.Cookie.SameSite;
import ej.hoka.http.encoding.ContentEncoding;
import ej.hoka.http.encoding.EncodingRegistry;
import ej.hoka.log.HokaLogger;

/**
 * Represents a HTTP Response.
 */
public class HttpResponse {

	/**
	 * The colon character.
	 */
	private static final String RESPONSE_COLON = ": "; //$NON-NLS-1$

	/**
	 * The HTTP/1.1 version String.
	 */
	private static final String RESPONSE_HTTP11 = "HTTP/1.1 "; //$NON-NLS-1$

	/**
	 * The Content-Type: String.
	 */
	private static final String RESPONSE_CONTENTTYPE = HttpConstants.HEADER_CONTENT_TYPE + RESPONSE_COLON;

	/**
	 * The status.
	 */
	private String status;

	/**
	 * The mime type.
	 */
	private String mimeType;

	/**
	 * Unique field to store the data object to be used, it can be either a byte[] or an InputStream.
	 */
	private Object data;

	/**
	 * Do not update this value by hand, use {@link #setLength(long)} to maintain HTTP header.
	 */
	private long length = -1; // -1 means unknown

	/**
	 * HTTP Response headers.
	 */
	private final Map<String, String> headers = new HashMap<>(5);

	/**
	 * HTTP Response cookies
	 */
	private Cookie[] cookies = new Cookie[0];

	/**
	 * Creates an empty {@link HttpResponse}.
	 */
	public HttpResponse() {
		setStatus(HttpConstants.HTTP_STATUS_OK);
		setData((String) null);
	}

	/**
	 * Adds a response header field.
	 *
	 * For Cookies @see {@link HttpResponse#addCookie(String, String, String, String, int, boolean, boolean)}
	 *
	 * @param name
	 *            name of the header field to set.
	 * @param value
	 *            value of the header filed.
	 */
	public void addHeader(String name, String value) {
		this.headers.put(name, value);
	}

	/**
	 * Adds a map of header fields.
	 *
	 * For Cookies @see {@link HttpResponse#addCookie(String, String, String, String, int, boolean, boolean)}
	 *
	 * @param map
	 *            map of headers to add
	 */
	public void addHeaders(Map<String, String> map) {
		this.headers.putAll(map);
	}

	/**
	 * Returns the header field value associated to the given header field <code>key</code>.
	 *
	 * @param key
	 *            a header field name (if <code>null</code>, <code>null</code> is returned).
	 * @return the replied header field value, <code>null</code> if the header field is not found.
	 */
	public String getHeader(String key) {
		if (key == null) {
			return null;
		}
		return this.headers.get(key.toLowerCase());
	}

	/**
	 * Returns the length (in bytes) of the response data or <code>-1</code> if the length is unknown.
	 *
	 * @return the length (in bytes) of the response data.
	 */
	protected long getLength() {
		return this.length;
	}

	/**
	 * Returns the MIME-TYPE of the response.
	 *
	 * @return the response MIME-TYPE.
	 */
	public String getMimeType() {
		return this.mimeType;
	}

	/**
	 * Returns the response status.
	 *
	 * @return the response status.
	 */
	public String getStatus() {
		return this.status;
	}

	/**
	 * Set the data contained by this response.
	 *
	 * @param data
	 *            the response data as a byte array (set to empty if <code>null</code> is given)
	 */
	public void setData(String data) {
		byte[] result;
		if (data == null) {
			result = new byte[] {};
		} else {
			result = data.getBytes();
		}
		this.data = result;
		setLength(result.length);
	}

	/**
	 * Set {@link HttpResponse} body using the {@link String} <code>data</code> as response data and the
	 * <code>encoding</code>.
	 *
	 * @param data
	 *            the {@link String} to be used as response body.
	 * @param encoding
	 *            the encoding used to transform the {@link String} <code>data</code> to bytes. The following encodings
	 *            can be used:
	 *            <ul>
	 *            <li><code>ISO-8859-1</code> ISO-8859-1 encoding, always supported by the platform
	 *            <li><code>UTF-8</code> UTF-8 encoding, only supported if the "Embed UTF-8 encoding" option is enabled
	 *            in the Run Configurations. If this option is not set, an {@link UnsupportedEncodingException} is
	 *            thrown.
	 *            <li><code>US-ASCII</code> US-ASCII encoding
	 *            </ul>
	 * @throws UnsupportedEncodingException
	 *             when the specified encoding is not supported.
	 */
	public void setData(String data, String encoding) throws UnsupportedEncodingException {
		byte[] result;
		if (data == null) {
			result = new byte[] {};
		} else {
			result = data.getBytes(encoding);
		}
		this.data = result;
		setLength(result.length);
	}

	/**
	 * Set the data contained by this response.
	 *
	 * @param data
	 *            the response data as a byte array (set to empty if <code>null</code> is given)
	 */
	public void setData(byte[] data) {
		byte[] result;
		if (data == null) {
			result = new byte[] {};
		} else {
			result = data;
		}
		this.data = result;
		setLength(result.length);
	}

	/**
	 * Sets the {@link InputStream} from which the response data can be read.
	 * <p>
	 * This method should be used only if response data length is not known in advance. If the length is known by
	 * advance the {@link #setData(InputStream, long)} should be used instead of this one. When response data is
	 * specified with this method, the response must be sent using the chunked transfer-coding which increase the
	 * response message size.
	 *
	 * @param dataStream
	 *            the {@link InputStream} from which the response data can be read, the stream will be closed
	 *            automatically when the response is sent.
	 */
	public void setData(InputStream dataStream) {
		setData(dataStream, -1);
	}

	/**
	 * Sets the {@link InputStream} from which the response data can be read.
	 * <p>
	 * This method should be used when response data length is known in advance. It allows to transfer response body
	 * without using the chunked transfer-coding. This reduces response message size.
	 *
	 * @param dataStream
	 *            the {@link InputStream} from which the response data can be read, the stream will be closed
	 *            automatically when the response is sent.
	 * @param length
	 *            the number of byte to be read from the {@link InputStream}.
	 */
	public void setData(InputStream dataStream, long length) {
		this.data = dataStream;
		setLength(length);
	}

	/**
	 * Sets the length of the response.
	 *
	 * @param length
	 *            the length of the response, if negative, the "content-length" field is removed.
	 */
	private void setLength(long length) {
		if (length < 0) {
			this.headers.remove(HttpConstants.HEADER_CONTENT_LENGTH);
		} else {
			this.headers.put(HttpConstants.HEADER_CONTENT_LENGTH, Long.toString(length));
		}
		this.length = length;
	}

	/**
	 * Set the response MIME-TYPE.
	 *
	 * @param mimeType
	 *            the response MIME-TYPE to set.
	 */
	public final void setMimeType(String mimeType) {
		this.mimeType = mimeType;
	}

	/**
	 * Set the response status.
	 *
	 * @param status
	 *            the response status to set. Should be one of the <code>HTTP_STATUS_*</code> constants defined in
	 *            {@link HttpConstants}
	 */
	public final void setStatus(String status) {
		this.status = status;
	}

	/**
	 * Adds the specified cookie to the response. This method can be called multiple times to set more than one cookie.
	 *
	 * @param name
	 *            name of the cookie
	 * @param value
	 *            value of the cookie
	 *
	 */
	public void addCookie(String name, String value) {
		addCookie(name, value, 0);
	}

	/**
	 * Adds cookie to the response. Can be invoked multiple times to insert more than one cookie.
	 *
	 * @param name
	 *            name of the cookie
	 * @param value
	 *            value of the cookie
	 * @param maxAge
	 *            max age of the cookie in seconds (negative for the not persistent cookie, zero - deletes the cookie)
	 */
	public void addCookie(String name, String value, int maxAge) {
		this.cookies = ArrayTools.add(this.cookies, new Cookie(name, value, maxAge));
	}

	/**
	 * Adds cookie to the response. Can be invoked multiple times to insert more than one cookie.
	 *
	 * @param name
	 *            name of the cookie
	 * @param value
	 *            value of the cookie
	 * @param maxAge
	 *            max age of the cookie in seconds (negative for the not persistent cookie, zero - deletes the cookie)
	 * @param secured
	 *            if true : cookie will be secured
	 * @param httpOnly
	 *            if true: cookie will be marked as http only
	 */
	public void addCookie(String name, String value, int maxAge, boolean secured, boolean httpOnly) {
		addCookie(name, value, null, null, maxAge, secured, httpOnly);
	}

	/**
	 * Adds the specified cookie to the response. This method can be called multiple times to set more than one cookie.
	 *
	 * @param name
	 *            name of the cookie
	 * @param value
	 *            value of the cookie
	 * @param domain
	 *            domain of the cookie
	 * @param path
	 *            path of the cookie
	 *
	 * @param maxAge
	 *            max age of the cookie in seconds (negative for the not persistent cookie, zero - deletes the cookie)
	 * @param secured
	 *            if true : cookie will be secured
	 * @param httpOnly
	 *            if true: cookie will be marked as http only
	 */
	public void addCookie(String name, String value, String domain, String path, int maxAge, boolean secured,
			boolean httpOnly) {
		addCookie(name, value, domain, path, maxAge, null, secured, httpOnly);
	}

	/**
	 * Adds the specified cookie to the response. This method can be called multiple times to set more than one cookie.
	 *
	 * @param name
	 *            name of the cookie
	 * @param value
	 *            value of the cookie
	 * @param domain
	 *            domain of the cookie
	 * @param path
	 *            path of the cookie
	 *
	 * @param maxAge
	 *            max age of the cookie in seconds (negative for the not persistent cookie, zero - deletes the cookie)
	 * @param expires
	 *            The maximum lifetime of the cookie as an HTTP-date timestamp.
	 * @param secured
	 *            if true : cookie will be secured
	 * @param httpOnly
	 *            if true: cookie will be marked as http only
	 */
	public void addCookie(String name, String value, String domain, String path, int maxAge, Date expires,
			boolean secured, boolean httpOnly) {
		addCookie(name, value, domain, path, maxAge, expires, secured, httpOnly, null);
	}

	/**
	 * Adds the specified cookie to the response. This method can be called multiple times to set more than one cookie.
	 *
	 * @param name
	 *            name of the cookie
	 * @param value
	 *            value of the cookie
	 * @param domain
	 *            domain of the cookie
	 * @param path
	 *            path of the cookie
	 *
	 * @param maxAge
	 *            max age of the cookie in seconds (negative for the not persistent cookie, zero - deletes the cookie)
	 * @param expires
	 *            The maximum lifetime of the cookie as an HTTP-date timestamp.
	 * @param secured
	 *            if true : cookie will be secured
	 * @param httpOnly
	 *            if true: cookie will be marked as http only
	 * @param sameSite
	 *            same site value of the cookie {@link SameSite}
	 */
	public void addCookie(String name, String value, String domain, String path, int maxAge, Date expires,
			boolean secured, boolean httpOnly, SameSite sameSite) {
		this.cookies = ArrayTools.add(this.cookies,
				new Cookie(name, value, domain, path, maxAge, expires, secured, httpOnly, sameSite));
	}

	/**
	 * Sends the {@link HttpResponse} to the {@link OutputStream}.
	 * <p>
	 * If the data of this response is an {@link InputStream}, closes it.
	 *
	 * @throws IOException
	 *
	 */
	/* default */ void sendResponse(OutputStream outputStream, ContentEncoding encodingHandler,
			EncodingRegistry encodingRegistry, int bufferSize) throws IOException {
		if (encodingHandler != null) {
			addHeader(HttpConstants.HEADER_CONTENT_ENCODING, encodingHandler.getId());
		}

		long length = getLength();

		if (length < 0) {
			// data will be transmitted using chunked transfer coding
			// only when dataStream is used, the size is known otherwise
			addHeader(HttpConstants.HEADER_TRANSFER_ENCODING, encodingRegistry.getChunkedTransferEncoding().getId());
		} // else the length is already defined in a header by the response

		writeHTTPHeader(outputStream);

		Object data = this.data;
		// only one of the next data can be defined.
		// A better way may be to specialize HTTPResponse for Raw String and
		// InputStream
		// and makes theses classes "visitable" by a HTTPWriter which is able to
		// visit both Raw String and InputStream HTTP Response
		// we keep this implementation to avoid new hierarchy for performance
		// but if the specialization evolves to a more and more
		// specific way, do it!
		if (data instanceof byte[]) {
			byte[] dataArray = (byte[]) data;
			sendRawDataResponse(dataArray, outputStream, encodingHandler, encodingRegistry);
		} else if (data != null) {
			try (InputStream dataStream = (InputStream) data) {
				sendInputStreamResponse(dataStream, outputStream, encodingHandler, encodingRegistry, bufferSize);
			}
		}

		outputStream.flush();
	}

	private void sendRawDataResponse(byte[] rawData, OutputStream outputStream, ContentEncoding encodingHandler,
			EncodingRegistry encodingRegistry) throws IOException {
		try (OutputStream dataOutput = encodingRegistry.getIdentityTransferCodingHandler().open(this, outputStream)) {
			if (encodingHandler != null) {
				try (OutputStream encodedDataOutput = encodingHandler.open(dataOutput)) {
					writeAndFlush(rawData, encodedDataOutput);
				}
			} else {
				writeAndFlush(rawData, dataOutput);
			}
		}
	}

	private void sendInputStreamResponse(InputStream dataStream, OutputStream outputStream,
			ContentEncoding encodingHandler, EncodingRegistry encodingRegistry, int bufferSize) {
		try (OutputStream dataOutput = (this.length == -1)
				? encodingRegistry.getChunkedTransferEncoding().open(this, outputStream)
				: encodingRegistry.getIdentityTransferCodingHandler().open(this, outputStream)) {
			try (OutputStream ecodedOutput = (encodingHandler != null) ? encodingHandler.open(dataOutput) : null) {
				final OutputStream output = (ecodedOutput != null) ? ecodedOutput : dataOutput;
				final byte[] readBuffer = new byte[bufferSize];
				while (true) {
					int len = dataStream.read(readBuffer);

					if (len < 0) { // read until EOF is reached
						break;
					}
					// store read data
					output.write(readBuffer, 0, len);
					output.flush();
				}
			}
		} catch (Throwable t) {
			HokaLogger.instance.error(t);
		}
	}

	/**
	 * Writes the HTTP Header using the {@link OutputStream} <code>output</code>.
	 *
	 * @param output
	 *            {@link OutputStream}
	 * @throws IOException
	 *             when the connection is lost
	 */
	private void writeHTTPHeader(OutputStream output) throws IOException {
		final byte[] eofHeader = HttpConstants.END_OF_LINE.getBytes();

		output.write(RESPONSE_HTTP11.getBytes());
		output.write(getStatus().trim().getBytes());
		output.write(eofHeader);

		if (this.mimeType != null) {
			output.write(RESPONSE_CONTENTTYPE.getBytes());
			output.write(this.mimeType.getBytes());
			output.write(eofHeader);
		}

		// add header parameters
		for (Entry<String, String> entry : this.headers.entrySet()) {
			String key = entry.getKey();
			String value = entry.getValue();
			output.write(key.getBytes());
			output.write(RESPONSE_COLON.getBytes());
			output.write(value.getBytes());
			output.write(eofHeader);
		}

		for (Cookie cookie : this.cookies) {
			output.write("Set-Cookie".getBytes()); //$NON-NLS-1$
			output.write(RESPONSE_COLON.getBytes());
			output.write(cookie.toString().getBytes());
			output.write(eofHeader);
		}

		output.write(eofHeader);
	}

	private static void writeAndFlush(byte[] data, OutputStream stream) throws IOException {
		stream.write(data);
		stream.flush();
		stream.close();
	}

}