/*
 * Copyright 2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microvg.bvi;

import java.awt.geom.GeneralPath;

import ej.microui.display.MicroUIGraphicsContext;
import ej.microui.display.MicroUIImageFormat;
import ej.microui.display.UIDrawing;
import ej.microvg.LLVGEngine;
import ej.microvg.VGDrawing;
import ej.microvg.paint.VGPaint;

/**
 * A buffered vector image drawer is able to draw the shapes from MicroUI or MicroVG in a buffered vector image.
 * <p>
 * By default, it does not support any drawing. It must be overridden to manage GPU specificities.
 * <p>
 * The {@link BufferedVectorImageDrawer} available in {@link LLVGEngine#getBufferedVectorImageDrawer()} can be used to
 * draw the different shapes.
 */
public class BufferedVectorImageDrawer implements UIDrawing, VGDrawing {

	@Override
	public MicroUIImageFormat handledFormat() {
		return LLVGEngine.BUFFERED_VECTOR_IMAGE_FORMAT;
	}

	@Override
	public void drawPath(MicroUIGraphicsContext gc, GeneralPath path, float x, float y, float[] matrix, int fillRule,
			int blend, VGPaint paint) {
		// MicroVG drawings are enabled by default
		getDefaultDrawer().drawPath(gc, path, x, y, matrix, fillRule, blend, paint);
	}

	/**
	 * Gets the default buffered vector image drawer able to draw all MicroUI and MicroVG drawings in a buffered vector
	 * image.
	 *
	 * @return the buffered vector image drawer
	 */
	protected BufferedVectorImageDrawer getDefaultDrawer() {
		return LLVGEngine.Instance.getBufferedVectorImageDrawer();
	}

}
