/*
 * Java
 *
 * Copyright 2021-2022 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;

/**
 * Tests the linear gradient.
 */
public class TestGradient {

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Tests an horizontal gradient.
	 */
	@Test
	public static void testGradientHorizontal() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 0, g, 0xffff00); //$NON-NLS-1$
		TestUtilities.check("right", 99, 0, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests an horizontal gradient with stops.
	 */
	@Test
	public static void testGradientHorizontalStops() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0f, 0.25f, 1f });
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 25, 0, g, 0xffff00); //$NON-NLS-1$
		TestUtilities.check("right", 99, 0, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests a diagonal gradient.
	 */
	@Test
	public static void testGradientDiagonal() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 99, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.requestFlush();

		TestUtilities.check("top left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("bottom right", 99, 99, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests a translated horizontal gradient.
	 */
	@Test
	public static void testGradientHorizontalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(50, 50);
		path.lineTo(150, 50);
		path.lineTo(150, 150);
		path.lineTo(50, 150);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setTranslate(50, 50);
		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.requestFlush();

		TestUtilities.check("left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 50, g, 0xffff00); //$NON-NLS-1$
		TestUtilities.check("right", 149, 50, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests an horizontal gradient in a translated path.
	 */
	@Test
	public static void testGradientHorizontalGlobalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.requestFlush();

		TestUtilities.check("left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 50, g, 0xffff00); //$NON-NLS-1$
		TestUtilities.check("right", 149, 50, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests a scaled horizontal gradient in a translated path.
	 */
	@Test
	public static void testGradientHorizontalScaleGlobalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 49, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setScale(2, 1);

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.requestFlush();

		TestUtilities.check("left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 50, g, 0xffff00); //$NON-NLS-1$
		TestUtilities.check("right", 149, 50, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests a translated horizontal gradient in a scaled path.
	 */
	@Test
	public static void testGradientHorizontalTranslateGlobalScale() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(50, 0, 149, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setTranslate(-50, 0);

		Matrix matrix = new Matrix();
		matrix.setScale(2, 1);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("center", 99, 0, g, 0xffff00); //$NON-NLS-1$
		TestUtilities.check("right", 199, 0, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests a rotated horizontal gradient in a translated path.
	 */
	@Test
	public static void testGradientHorizontalRotateGlobalTranslate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 140, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setRotate(45);

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.requestFlush();

		TestUtilities.check("left", 50, 50, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 149, 149, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests a scaled horizontal gradient in a translated and rotated path.
	 */
	@Test
	public static void testGradientHorizontalScaleGlobalRotate() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 49, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff },
				new float[] { 0.0f, 0.5f, 1.0f });
		Matrix gradientMatrix = gradient.getMatrix();
		gradientMatrix.setScale(2, 2);

		Matrix matrix = new Matrix();
		matrix.setTranslate(50, 50);
		matrix.preRotate(45);

		VectorGraphicsPainter.fillGradientPath(g, path, matrix, gradient);
		display.requestFlush();

		TestUtilities.check("left", 50, 51, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("right", 50, 189, g, 0xffffff); //$NON-NLS-1$
	}

	/**
	 * Tests an horizontal gradient with a fully transparent alpha.
	 */
	@Test
	public static void testGradientColorTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient, FillType.WINDING, 0x00,
				BlendMode.SRC_OVER);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 0, g, 0x000000); //$NON-NLS-1$
		TestUtilities.check("right", 99, 0, g, 0x000000); //$NON-NLS-1$
	}

	/**
	 * Tests an horizontal gradient drawn with a partially transparent alpha.
	 */
	@Test
	public static void testGradientColorSemiTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0xffff0000, 0xffffff00, 0xffffffff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient, FillType.WINDING, 0x88,
				BlendMode.SRC_OVER);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0x800000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 0, g, 0x808000); //$NON-NLS-1$
		TestUtilities.check("right", 99, 0, g, 0x808080); //$NON-NLS-1$
	}

	/**
	 * Tests a partially transparent horizontal gradient drawn with a partially transparent alpha.
	 */
	@Test
	public static void testGradientSemiTransparentColorSemiTransparent() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(0, 0, 99, 0, new int[] { 0x88ff0000, 0x88ffff00, 0x88ffffff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient, FillType.WINDING, 0x88,
				BlendMode.SRC_OVER);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0x490000); //$NON-NLS-1$
		TestUtilities.check("center", 49, 0, g, 0x494900); //$NON-NLS-1$
		TestUtilities.check("right", 99, 0, g, 0x494949); //$NON-NLS-1$
	}

	/**
	 * Tests a gradient with the two points at the same place.
	 */
	@Test
	public static void testGradientTwoPointsAtSamePlace() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineTo(100, 0);
		path.lineTo(100, 100);
		path.lineTo(0, 100);
		path.close();

		LinearGradient gradient = new LinearGradient(49, 49, 49, 49, new int[] { 0xffff0000, 0xff880088, 0xff0000ff });

		VectorGraphicsPainter.fillGradientPath(g, path, new Matrix(), gradient);
		display.requestFlush();

		TestUtilities.check("left", 0, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("centerleft", 48, 0, g, 0xff0000); //$NON-NLS-1$
		TestUtilities.check("centerright", 50, 0, g, 0x0000ff); //$NON-NLS-1$
		TestUtilities.check("right", 99, 0, g, 0x0000ff); //$NON-NLS-1$
	}

	/**
	 * Tests a gradient path with clipping.
	 */
	@Test
	public static void testGradientWithClip() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();

		int x = 100;
		int y = 100;
		int width = 60;
		int height = 60;
		int testThickness = 1;

		int errors = 0;

		Path path = new Path();
		path.moveTo(0, 0);
		path.lineToRelative(0, height);
		path.lineToRelative(width, 0);
		path.lineToRelative(0, -height);
		path.close();

		// Gradient with same color at start and end
		LinearGradient gradient = new LinearGradient(0, 0, width, 0, new int[] { 0xff0000ff, 0xff0000ff });

		Matrix m = new Matrix();
		m.setTranslate(x, y); // Brings top left to x,y

		// Clip larger than path
		int outterClip = 10;
		g.setClip(x - outterClip, y - outterClip, width + 2 * outterClip, height + 2 * outterClip);

		g.setColor(0x0000ff);
		VectorGraphicsPainter.fillGradientPath(g, path, m, gradient);
		display.requestFlush();

		errors += TestUtilities.checkPeripheralArea("clip over inside", 0x0000ff, x + testThickness, y + testThickness,
				width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip over outside", 0x000000, x, y, width, height, testThickness,
				0, false);

		// Clip fit path
		x += 100;
		m.postTranslate(100, 0);

		outterClip = 0;
		g.setClip(x - outterClip, y - outterClip, width + 2 * outterClip, height + 2 * outterClip);

		g.setColor(0x0000ff);
		VectorGraphicsPainter.fillGradientPath(g, path, m, gradient);
		display.requestFlush();

		errors += TestUtilities.checkPeripheralArea("clip fit inside", 0x0000ff, x + testThickness, y + testThickness,
				width - 2 * testThickness, height - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip fit outside", 0x000000, x, y, width, height, testThickness, 0,
				false);
		// Clip thinner than path
		x += 100;
		m.postTranslate(100, 0);

		outterClip = -10;
		g.setClip(x - outterClip, y - outterClip, width + 2 * outterClip, height + 2 * outterClip);

		g.setColor(0x0000ff);
		VectorGraphicsPainter.fillGradientPath(g, path, m, gradient);
		display.requestFlush();
		errors += TestUtilities.checkPeripheralArea("clip inner inside", 0x0000ff, x - outterClip + testThickness,
				y - outterClip + testThickness, width + 2 * outterClip - 2 * testThickness,
				height + 2 * outterClip - 2 * testThickness, testThickness, 0, false);
		errors += TestUtilities.checkPeripheralArea("clip inner outside", 0x000000, x - outterClip, y - outterClip,
				width + 2 * outterClip, height + 2 * outterClip, testThickness, 0, false);
		assertEquals(0, errors);
	}

}
