/*
 * Java
 *
 * Copyright 2010-2019 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

import ej.annotation.Nullable;
import ej.microui.util.EventHandler;

/**
 * A <code>Display</code> object represents a pixelated screen in the platform, and there is a display for each such
 * pixelated screen. Available displays can be retrieved with the method {@link #getAllDisplays()}. A default display is
 * defined in every MicroUI implementation and can be fetched with the method {@link #getDefaultDisplay()}.<br>
 * <br>
 * A display is able to render a <code>Displayable</code> on its implementation screen. Only one
 * <code>Displayable</code> can be set on a display at a time; it is said to be visible or to be shown. The visible
 * <code>Displayable</code> can be retrieved with the method {@link #getDisplayable()}.<br>
 * <br>
 * {@link Displayable#show()} allows the Displayable to be selected for rendering on its display. It can be called at
 * any time by the application, for instance in response to user inputs.<br>
 * <br>
 * <code>Display</code> uses a {@link GraphicsContext} to draw on its corresponding screen. All draw actions are
 * serialized. The application should not use a display's graphics context outside the events mechanism
 * <code>repaint()</code> and <code>paint()</code>. Nevertheless, for exceptional cases a new
 * <code>GraphicsContext</code> may be created using <code>getNewGraphicsContext()</code>. This new <code>
 * GraphicsContext</code> bypasses the standard serialized drawing mechanism and allows drawings to be rendered on the
 * display at any time.<br>
 * <br>
 * All events on a display are serialized: repaint, callSerially, handleEvent etc. A display uses a <code>
 * FIFOPump</code> to manage its serialized event mechanism.
 */
public class Display {

	/**
	 * Returns the display's event serializer or <code>null</code>.
	 *
	 * @return the display's event serializer or <code>null</code>.
	 * @since 2.0
	 */
	public EventHandler getEventSerializer() {
		throw new RuntimeException();
	}

	/**
	 * Returns the height in pixels of the display screen area available to the application.
	 *
	 * @return height of the display screen area.
	 */
	public int getHeight() {
		throw new RuntimeException();
	}

	/**
	 * Returns the width in pixels of the display screen area available to the application.
	 *
	 * @return width of the display screen area.
	 */
	public int getWidth() {
		throw new RuntimeException();
	}

	/**
	 * Returns the number of bits per pixel of the display.
	 *
	 * @return the number of bits per pixel
	 */
	public int getBPP() {
		throw new RuntimeException();
	}

	/**
	 * Tells whether the display offers color.
	 *
	 * @return if display has color
	 */
	public boolean isColor() {
		throw new RuntimeException();
	}

	/**
	 * Gets the number of colors that can be represented on the device.<br>
	 * Note that the number of colors for a black and white display is 2.
	 *
	 * @return the number of colors
	 */
	public int getNumberOfColors() {
		throw new RuntimeException();
	}

	/**
	 * Gets the number of alpha transparency levels supported by the implementation.<br>
	 * <br>
	 * The minimum possible is 2, which represents full transparency and full opacity with no blending. If the return
	 * value is greater than 2, the implementation manages blending.
	 *
	 * @return the number of alpha levels
	 */
	public int getNumberOfAlphaLevels() {
		throw new RuntimeException();
	}

	/**
	 * Returns if the display uses an underlying double buffer (either hardware or software). This technique is useful
	 * to avoid flickering while the user is drawing.
	 *
	 * @return true if and only if a double buffer is used for the display
	 */
	public boolean isDoubleBuffered() {
		throw new RuntimeException();
	}

	/**
	 * Gets the color that will be displayed if the specified color is requested.<br>
	 * For example, with a monochrome display, this method will return either 0xFFFFFF (white) or 0x000000 (black)
	 * depending on the brightness of the specified color.
	 *
	 * @param color
	 *            the desired color in 0x00RRGGBB format.
	 * @return the corresponding color that will be displayed on the graphics context (in 0x00RRGGBB format).
	 */
	public final int getDisplayColor(int color) {
		throw new RuntimeException();
	}

	/**
	 * Tells whether the display has backlight.
	 *
	 * @return if display has backlight
	 */
	public boolean hasBacklight() {
		throw new RuntimeException();
	}

	/**
	 * Sets the contrast of the display. <code>contrast</code> value range is 0-100
	 *
	 * @param contrast
	 *            the new value of the contrast
	 */
	public void setContrast(int contrast) {
		throw new RuntimeException();
	}

	/**
	 * Returns the contrast of the display.
	 *
	 * @return the current contrast of the display (range 0-100)
	 */
	public int getContrast() {
		throw new RuntimeException();
	}

	/**
	 * Switches on or off the backlight of the display.
	 *
	 * @param on
	 *            Switch on the backlight if <code>true</code>; switch off the backlight if <code>false</code>
	 */
	public void switchBacklight(boolean on) {
		throw new RuntimeException();
	}

	/**
	 * Sets the backlight of the display. <code>backlight</code> value range is 0-100
	 *
	 * @param backlight
	 *            the new value of the backlight
	 */
	public void setBacklight(int backlight) {
		throw new RuntimeException();
	}

	/**
	 * Returns the current backlight setting
	 *
	 * @return the current backlight setting (range 0-100)
	 */
	public int getBacklight() {
		throw new RuntimeException();
	}

	/**
	 * Sets the current backlight color, if it is allowed by implementation.
	 *
	 * @param rgbColor
	 *            the color to set
	 */
	public void setBacklightColor(int rgbColor) {
		throw new RuntimeException();
	}

	/**
	 * Returns the current backlight color. Returned value is interpreted as a 24-bit RGB color, where the eight less
	 * significant bits matches the blue component, the next eight bits matches the green component and the next eight
	 * bits matches the red component. By default, this method returns 0xFFFFFF (white) and sub-classes should overwrite
	 * this default behavior.
	 *
	 * @return the color of the backlight
	 */
	public int getBacklightColor() {
		throw new RuntimeException();
	}

	/**
	 * Returns all available displays. It is never <code>null</code> but the array may be empty.
	 *
	 * @return all available displays.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 */
	public static Display[] getAllDisplays() throws IllegalStateException {
		throw new RuntimeException();
	}

	/**
	 * Returns the default display of the system. It can be null if there is no display. The notion of default display
	 * is defined by the implementation.<br>
	 *
	 * @return the default display or <code>null</code>.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 */
	public static @Nullable Display getDefaultDisplay() throws IllegalStateException {
		throw new RuntimeException();
	}

	/**
	 * Returns the current <code>Displayable</code> object in the <code>Display</code>.<br>
	 * The value returned by <code>getDisplayable()</code> may be <code>null</code> if no <code>Displayable</code> is
	 * visible.
	 *
	 * @return the current <code>Displayable</code> object in the <code>Display</code>
	 */
	public @Nullable Displayable getDisplayable() {
		throw new RuntimeException();
	}

	/**
	 * Returns a new <code>GraphicsContext</code> which works on the same system screen as this display. With this
	 * <code>GraphicsContext</code>, it is possible to draw on the system screen at any time without modifying the
	 * normal system execution. The new graphics context has its own clip, color, font etc. After each draw action (a
	 * drawLine for example), the system screen will show the drawn pixels. <br>
	 * If the normal system execution is repainting at the same time, the last draw action will be visible (the previous
	 * one will be hidden by the last one). It is not possible to determine which draw action will be done last.
	 *
	 * @return a new graphics context on the display
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new graphics context.
	 */
	public GraphicsContext getNewGraphicsContext() {
		throw new RuntimeException();
	}

	/**
	 * Returns a new <code>ExplicitFlush</code> which works on the same system screen as this display. With this
	 * <code>ExplicitFlush</code>, it is possible to draw on the system screen at any time without modifying the normal
	 * system execution. The new graphics context has its own clip, color, font etc. Each draw action will not be
	 * automatically flushed. The user has to flush it via the <code>ExplicitFlush.flush()</code> method.<br>
	 * If the normal system execution is repainting at the same time, the last unflushed draw actions will be visible
	 * (the previous one will be hidden by the last one). It is not possible to determine which draw action will be done
	 * last.
	 *
	 * @return a new graphics context with explicit flush on the display
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new graphics context.
	 */
	public ExplicitFlush getNewExplicitFlush() {
		throw new RuntimeException();
	}

	/**
	 * Creates an image with the full content of the display.
	 * <p>
	 * This call is identical to: <code>getScreenshot(0, 0, display.getWidth(), display.getHeight())</code>.
	 *
	 * @return the created image.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @see GraphicsContext#drawRegion(Display, int, int, int, int, int, int, int, int)
	 * @since 2.0
	 */
	public Image getScreenshot() {
		throw new RuntimeException();
	}

	/**
	 * Creates an image with the content of the display region specified thanks the given rectangle.
	 *
	 * @param x
	 *            the x coordinate of the upper-left corner of the region to copy.
	 * @param y
	 *            the y coordinate of the upper-left corner of the region to copy.
	 * @param width
	 *            the width of the region to copy.
	 * @param height
	 *            the height of the region to copy.
	 * @return the created image.
	 * @throws IllegalArgumentException
	 *             if the zone to copy is out of the bounds of the source image or if either <code>width</code> or
	 *             <code>height</code> is zero or negative.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @see GraphicsContext#drawRegion(Display, int, int, int, int, int, int, int, int)
	 * @since 2.0
	 */
	public Image getScreenshot(int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Serializes a call event in the system event stream. When the event is processed, the <code>run()</code> method of
	 * the <code>Runnable</code> object is called.<br>
	 * Multiple call events may be requested with <code>callSerially()</code>: they will occur in the order in which
	 * they were requested (first in first out policy).<br>
	 * The call to the <code>run()</code> method of the <code>Runnable</code> object is performed asynchronously
	 * Therefore <code>callSerially()</code> will never block waiting for the <code>run()</code> method to finish.<br>
	 * The <code>run()</code> method should return quickly, as with other callback methods.<br>
	 * The <code>callSerially()</code> mechanism may be used by applications as a synchronization tool in the event
	 * stream.
	 *
	 * @param run
	 *            a <code>Runnable</code> object to call
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to get the display.
	 */
	public void callSerially(@Nullable Runnable run) {
		throw new RuntimeException();
	}

	/**
	 * Sends <code>event</code> in the event stream and blocks the current thread (with all its locks) until the
	 * <code>event</code> processing is finished.
	 *
	 * @param event
	 *            the event to send and to wait for.
	 * @throws RuntimeException
	 *             if the current thread is the <code>Display</code>'s events thread.
	 */
	public void waitForEvent(int event) {
		throw new RuntimeException();
	}

	/**
	 * Blocks the current thread (with all its locks) until all events outstanding at the time of the call have been
	 * processed.
	 *
	 * @throws RuntimeException
	 *             if the current thread is the <code>Display</code>'s events thread.
	 */
	public void waitForEvent() {
		throw new RuntimeException();
	}

	/**
	 * Gets whether the given thread is the display events thread.
	 *
	 * @param thread
	 *            the thread to check
	 *
	 * @return <code>true</code> if the given thread is the display events thread, <code>false</code> otherwise.
	 * @since 2.0
	 */
	public boolean isDisplayThread(Thread thread) {
		throw new RuntimeException();
	}

	/**
	 * Gets whether the current thread is the display events thread.
	 *
	 * @return <code>true</code> if the current thread is the display events thread, <code>false</code> otherwise.
	 * @since 2.0
	 */
	public boolean isDisplayThread() {
		throw new RuntimeException();
	}

	/**
	 * Sets the priority of the display events processing.
	 *
	 * @param priority
	 *            the new priority of display events processing
	 * @throws IllegalArgumentException
	 *             If the priority is not in the range {@link Thread#MIN_PRIORITY} to {@link Thread#MAX_PRIORITY}.
	 */
	public void setPriority(int priority) {
		throw new RuntimeException();
	}

	/**
	 * Injects a MicroUI event to be handled by the event generator associated with this Display.
	 *
	 * @param event
	 *            an event in the MicroUI format
	 */
	public void handleEvent(int event) {
		throw new RuntimeException();
	}

}
