/*
 * Copyright 2016-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector.combinator;

import ej.annotation.Nullable;
import ej.mwt.stylesheet.selector.Selector;

/**
 * A combinator is a selector that aggregates two selectors.
 * <p>
 * The specificity of a combinator is the sum of the specificities of its selectors.
 */
public abstract class Combinator implements Selector {

	private final Selector firstSelector;
	private final Selector secondSelector;

	/**
	 * Creates a combinator.
	 *
	 * @param firstSelector
	 *            the first selector.
	 * @param secondSelector
	 *            the second selector.
	 */
	public Combinator(Selector firstSelector, Selector secondSelector) {
		this.firstSelector = firstSelector;
		this.secondSelector = secondSelector;
	}

	/**
	 * Returns the first selector of this combinator.
	 *
	 * @return the first selector.
	 */
	public Selector getFirstSelector() {
		return this.firstSelector;
	}

	/**
	 * Returns the second selector of this combinator.
	 *
	 * @return the second selector.
	 */
	public Selector getSecondSelector() {
		return this.secondSelector;
	}

	@Override
	public int getSpecificity() {
		return this.firstSelector.getSpecificity() + this.secondSelector.getSpecificity();
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && obj.getClass() == getClass()) {
			Combinator combinator = (Combinator) obj;
			return this.firstSelector.equals(combinator.firstSelector)
					&& this.secondSelector.equals(combinator.secondSelector);
		}
		return false;
	}

	@Override
	public int hashCode() {
		return this.firstSelector.hashCode() + 7 * this.secondSelector.hashCode();
	}
}
