/*
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.android.internal;

import android.content.Context;
import android.graphics.Bitmap;
import android.graphics.Rect;

/**
 * This interface can be used by Android Implementations of Foundation Libraries in order to get the current state of
 * the MicroEJ support engine and to flush the display.
 */
public interface InternalSupport {

	/**
	 * Checks whether the MicroEJ support engine has been started.
	 *
	 * @throws IllegalStateException
	 *             if the engine has not been started.
	 * @see com.microej.android.application.ApplicationSupport#startup()
	 */
	void checkStarted();

	/**
	 * Returns the Android application context of the MicroEJ support engine.
	 *
	 * @return the context.
	 * @throws IllegalStateException
	 *             if the context has not been set.
	 * @see com.microej.android.application.ApplicationSupport#setContext(Context)
	 */
	Context getContext();

	/**
	 * Flushes the given back buffer on the application display.
	 * <p>
	 * This method is called by the front panel when a flush is requested by the application and the tearing signal has
	 * been received.
	 *
	 * @param bitmap
	 *            a bitmap containing the back buffer.
	 * @param bounds
	 *            the bounds of the bitmap to flush.
	 */
	void flush(Bitmap bitmap, Rect bounds);

	/**
	 * Sets the current display listener.
	 *
	 * @param displayListener
	 *            the listener to set.
	 */
	void setDisplayListener(InternalDisplayListener displayListener);

	/**
	 * Unsets the current display listener.
	 * <p>
	 * This method does nothing if the current display listener is not the given object.
	 *
	 * @param displayListener
	 *            the listener to unset.
	 */
	void unsetDisplayListener(InternalDisplayListener displayListener);
}
