/*
 * Copyright 2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.util.debug;

import ej.mwt.Container;
import ej.mwt.Widget;

/**
 * Provides helpers to analyze a hierarchy of widgets.
 */
public class HierarchyInspector {

	// Prevents initialization.
	private HierarchyInspector() {
	}

	/**
	 * Counts the number of containers in the child hierarchy of a widget.
	 * <p>
	 * The method is recursive: if the given widget is a container, it browses its children.
	 *
	 * @param widget
	 *            the root widget.
	 * @return the number of containers.
	 */
	public static int countNumberOfContainers(Widget widget) {
		int number = 0;
		if (widget instanceof Container) {
			number++;
			Container container = (Container) widget;
			int numChildren = container.getChildrenCount();
			for (int i = 0; i < numChildren; i++) {
				Widget child = container.getChild(i);
				number += countNumberOfContainers(child);
			}
		}
		return number;
	}

	/**
	 * Counts the number of widgets in the child hierarchy of a widget.
	 * <p>
	 * The method is recursive: if the given widget is a container, it browses its children.
	 *
	 * @param widget
	 *            the root widget.
	 * @return the number of containers.
	 */
	public static int countNumberOfWidgets(Widget widget) {
		int number = 0;
		if (widget instanceof Container) {
			Container container = (Container) widget;
			int numChildren = container.getChildrenCount();
			for (int i = 0; i < numChildren; i++) {
				Widget child = container.getChild(i);
				number += countNumberOfWidgets(child);
			}
		} else {
			number++;
		}
		return number;
	}

	/**
	 * Counts the maximum depth of the child hierarchy of a widget.
	 * <p>
	 * The method is recursive: if the given widget is a container, it browses its children.
	 *
	 * @param widget
	 *            the root widget.
	 * @return the maximum depth.
	 */
	public static int countMaxDepth(Widget widget) {
		return countMaxDepth(widget, 0);
	}

	private static int countMaxDepth(Widget widget, int currentDepth) {
		int maxDepth = currentDepth;
		if (widget instanceof Container) {
			Container container = (Container) widget;
			int numChildren = container.getChildrenCount();
			for (int i = 0; i < numChildren; i++) {
				Widget child = container.getChild(i);
				int childDepth = countMaxDepth(child, currentDepth + 1);
				maxDepth = Math.max(maxDepth, childDepth);
			}
		}
		return maxDepth;
	}

}
