/*
 * Copyright 2023-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.microui.display;

/**
 * A rectangle with 4 coordinates.
 */
public class Rectangle {

	/**
	 * Denotes an empty rectangle. Prevents the use of null rectangles.
	 */
	public static final Rectangle NULL_RECTANGLE = new Rectangle(0, 0, -1, -1);

	private int x1;
	private int y1;
	private int x2;
	private int y2;

	/**
	 * Creates a rectangle specifying its 4 coordinates.
	 *
	 * @param x1
	 *            the left x coordinate
	 * @param y1
	 *            the top y coordinate
	 * @param x2
	 *            the right x coordinate
	 * @param y2
	 *            the bottom y coordinate
	 */
	public Rectangle(int x1, int y1, int x2, int y2) {
		this.x1 = x1;
		this.y1 = y1;
		this.x2 = x2;
		this.y2 = y2;
	}

	/**
	 * Creates a rectangle from another one.
	 *
	 * @param rectangle
	 *            the rectangle to copy
	 */
	public Rectangle(Rectangle rectangle) {
		this.x1 = rectangle.x1;
		this.y1 = rectangle.y1;
		this.x2 = rectangle.x2;
		this.y2 = rectangle.y2;
	}

	/**
	 * Gets the left x coordinate.
	 *
	 * @return the left x.
	 */
	public int getX1() {
		return this.x1;
	}

	/**
	 * Gets the top y coordinate.
	 *
	 * @return the top y.
	 */
	public int getY1() {
		return this.y1;
	}

	/**
	 * Gets the right x coordinate.
	 *
	 * @return the right x.
	 */
	public int getX2() {
		return this.x2;
	}

	/**
	 * Gets the bottom y coordinate.
	 *
	 * @return the bottom y.
	 */
	public int getY2() {
		return this.y2;
	}

	/**
	 * Sets the left x.
	 *
	 * @param x
	 *            the x to set.
	 */
	public void setX1(int x) {
		this.x1 = x;
	}

	/**
	 * Sets the top y.
	 *
	 * @param y
	 *            the y to set.
	 */
	public void setY1(int y) {
		this.y1 = y;
	}

	/**
	 * Sets the right x.
	 *
	 * @param x
	 *            the x to set.
	 */
	public void setX2(int x) {
		this.x2 = x;
	}

	/**
	 * Sets the bottom y.
	 *
	 * @param y
	 *            the y to set.
	 */
	public void setY2(int y) {
		this.y2 = y;
	}

	/**
	 * Gets the width of this rectangle.
	 *
	 * @return the width
	 */
	public int getWidth() {
		return this.x2 - this.x1 + 1;
	}

	/**
	 * Gets the height of this rectangle.
	 *
	 * @return the height
	 */
	public int getHeight() {
		return this.y2 - this.y1 + 1;
	}

	/**
	 * Checks whether a rectangle is included or equal to this rectangle.
	 *
	 * @param rectangleCandidate
	 *            the rectangle to check
	 * @return <code>true</code> if the given rectangle is included in this one, <code>false</code> otherwise
	 */
	public boolean contains(Rectangle rectangleCandidate) {
		return rectangleCandidate.x1 >= this.x1 && rectangleCandidate.y1 >= this.y1 && rectangleCandidate.x2 <= this.x2
				&& rectangleCandidate.y2 <= this.y2;
	}

	/**
	 * Tells whether a rectangle is equal to this rectangle.
	 *
	 * @param rectangleCandidate
	 *            the rectangle to check
	 * @return <code>true</code> if the given rectangle is equal to this one, <code>false</code> otherwise
	 */
	public boolean isEqual(Rectangle rectangleCandidate) {
		return rectangleCandidate.x1 == this.x1 && rectangleCandidate.y1 == this.y1 && rectangleCandidate.x2 == this.x2
				&& rectangleCandidate.y2 == this.y2;
	}

	/**
	 * Tells whether the rectangle is empty.
	 *
	 * @return <code>true</code> if the rectangle has a null area
	 */
	public boolean isEmpty() {
		return getWidth() <= 0 || getHeight() <= 0;
	}

	/**
	 * Gets the size of the rectangle.
	 *
	 * @return rectangle width x rectangle height
	 */
	public int getSize() {
		return getWidth() * getHeight();
	}
}
