/*
 * <Java>
 *
 * Copyright 2015-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.service.loader;

import java.io.IOException;
import java.io.InputStream;

import ej.annotation.Nullable;

/**
 * Service loader that retrieves services implementations following standard Java service definition.
 * <p>
 * Services are defined in files located in {@value #SERVICES_FOLDER} folder:
 * <ul>
 * <li>the file is named after the implemented service (fully qualified name),</li>
 * <li>the content of the file is the fully qualified name of the implementation class.</li>
 * </ul>
 */
public class MetaInfServiceLoader extends DependencyInjectionServiceLoader {

	/**
	 * Folder where the services are defined.
	 */
	public static final String SERVICES_FOLDER = "/META-INF/services/"; //$NON-NLS-1$

	/**
	 * Creates a new META-INF properties service loader.
	 */
	public MetaInfServiceLoader() {
		super();
	}

	@Override
	@Nullable
	protected String getImplementationName(String key) {
		String serviceFileName = SERVICES_FOLDER + key;
		try (InputStream inputStream = MetaInfServiceLoader.class.getResourceAsStream(serviceFileName)) {
			if (inputStream != null) {
				while (true) {
					// read line
					StringBuilder lineBuilder = new StringBuilder();
					while (true) {
						int readByte = inputStream.read();
						if (readByte == -1 || readByte == '\n') {
							// end of file or end of line
							break;
						} else {
							lineBuilder.append((char) readByte);
						}
					}
					String line = lineBuilder.toString().trim();
					if (!line.isEmpty() && line.charAt(0) != '#') {
						// read only first valid line
						return line;
					}
				}
			}
		} catch (IOException e) {
			// cannot read file
		}
		return null;
	}

}
