/*
 * <Java>
 *
 * Copyright 2014-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.service;

/**
 * An extension of service loader that allows to register services dynamically.
 */
public interface ServiceRegistry extends ServiceLoader {

	/**
	 * Registers a service instance.
	 * <p>
	 * The given instance is returned each time the service is requested by {@link #getService(Class)}.
	 * <p>
	 * If the service already has a registered instance, the previous one is replaced.
	 *
	 * @param service
	 *            the implemented service.
	 * @param instance
	 *            the service instance.
	 * @param <T>
	 *            the service type.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to register the given service instance.
	 */
	<T> void register(Class<T> service, T instance);

	/**
	 * Unregisters a service instance.
	 * <p>
	 * If the given instance is not the one currently registered for the given service, nothing is done.
	 *
	 * @param service
	 *            the implemented service.
	 * @param instance
	 *            the service instance.
	 * @param <T>
	 *            the service type.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to unregister the given service instance.
	 */
	<T> void unregister(Class<T> service, T instance);

}
