/*
 * Java
 *
 * Copyright 2024 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 */
package org.junit;

/**
 * <p>
 * A set of methods useful for stating assumptions about the conditions in which
 * a test is meaningful. A failed assumption does not mean the code is broken,
 * but that the test provides no useful information. Assume basically means
 * "don't run this test if these conditions don't apply". The default JUnit
 * runner skips tests with failing assumptions. Custom runners may behave
 * differently.
 * </p>
 *
 * <p>
 * A good example of using assumptions is in Theories where they are needed to
 * exclude certain datapoints that aren't suitable or allowed for a certain test
 * case. Failed assumptions are usually not logged, because there may be many
 * tests that don't apply to certain configurations.
 * </p>
 *
 * <p>
 * These methods can be used directly: Assume.assumeTrue(...), however, they
 * read better if they are referenced through static import:
 * </p>
 *
 * <pre>
 import static org.junit.Assume.*;
    ...
    assumeTrue(...);
 * </pre>
 */
public class Assume {

	private Assume() {
		// Private constructor to prevent instantiation
	}

	/**
	 * If called with an expression evaluating to <code>false</code>, the test will
	 * halt and be ignored.
	 *
	 * @param b
	 *            If <code>false</code>, the method will attempt to stop the test
	 *            and ignore it by throwing {@link AssumptionViolatedException}.
	 */
	public static void assumeTrue(boolean b) {
		if (!b) {
			throw new AssumptionViolatedException();
		}
	}

	/**
	 * If called with an expression evaluating to <code>false</code>, the test will
	 * halt and be ignored.
	 *
	 * @param message
	 *            A message to pass to {@link AssumptionViolatedException}.
	 * @param b
	 *            If <code>false</code>, the method will attempt to stop the test
	 *            and ignore it by throwing {@link AssumptionViolatedException}.
	 */
	public static void assumeTrue(String message, boolean b) {
		if (!b) {
			throw new AssumptionViolatedException(message);
		}
	}

	/**
	 * The inverse of {@link #assumeTrue(boolean)}.
	 */
	public static void assumeFalse(boolean b) {
		assumeTrue(!b);
	}

	/**
	 * The inverse of {@link #assumeTrue(String, boolean)}.
	 */
	public static void assumeFalse(String message, boolean b) {
		assumeTrue(message, !b);
	}
}
