/*
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.microej.android.application;

import java.lang.reflect.InvocationTargetException;

import com.microej.android.MicroEJSupport;

import android.app.Activity;
import android.os.Bundle;
import android.view.MotionEvent;
import android.view.View;
import android.view.View.OnTouchListener;

/**
 * This Activity implementation can run a MicroEJ Application on Android using the MicroEJ support engine.
 * <p>
 * The concrete class should implement {@link #getApplicationMainClass()} to provide the main class of the MicroEJ
 * Application. It can also override {@link #runApplication()} to run the Application differently.
 */
public abstract class MicroEJActivity extends Activity {

	private MicroEJView view;

	/**
	 * Returns the main class of the MicroEJ Application.
	 *
	 * @return the main class of the MicroEJ Application.
	 */
	protected abstract String getApplicationMainClass();

	/**
	 * Runs the MicroEJ Application.
	 * <p>
	 * This method invokes the main method of the Application main class (see {@link #getApplicationMainClass()}). This
	 * behavior can be changed by overriding this method.
	 */
	protected void runApplication() {
		String mainClassName = getApplicationMainClass();
		try {
			Class<?> mainClass = Class.forName(mainClassName);
			mainClass.getMethod("main", String[].class).invoke(null, (Object) new String[0]);
		} catch (InvocationTargetException e) {
			e.getCause().printStackTrace();
		} catch (ReflectiveOperationException e) {
			System.err.println("Error invoking main method of main class");
		}
	}

	@Override
	protected void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);

		ApplicationSupport applicationSupport = MicroEJSupport.getApplicationSupport();
		applicationSupport.startup();
		applicationSupport.setContext(this);

		this.view = new MicroEJView(this);
		this.view.setOnTouchListener(createOnTouchListener());

		setContentView(this.view);

		new Thread(new Runnable() {
			@Override
			public void run() {
				runApplication();
			}
		}).start();
	}

	private OnTouchListener createOnTouchListener() {
		return new OnTouchListener() {
			@Override
			public boolean onTouch(View view, MotionEvent event) {
				return handleTouchEvent(event);
			}
		};
	}

	private boolean handleTouchEvent(MotionEvent event) {
		switch (event.getAction()) {
		case MotionEvent.ACTION_DOWN:
			// touch press but can be canceled
			MicroEJSupport.getApplicationSupport().onPointerPress(this.view, (int) event.getX(), (int) event.getY());
			return true;
		case MotionEvent.ACTION_CANCEL:
			// touch canceled (touch event managed by system)
		case MotionEvent.ACTION_UP:
			// touch released
			MicroEJSupport.getApplicationSupport().onPointerRelease(this.view);
			return true;
		case MotionEvent.ACTION_MOVE:
			// touch move
			MicroEJSupport.getApplicationSupport().onPointerMove(this.view, (int) event.getX(), (int) event.getY());
			return true;
		default:
			return super.onTouchEvent(event);
		}
	}
}
