/*
 * Java
 *
 * Copyright 2008-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */

package ej.fp;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;

/**
 * A composite is a widget that can contain other {@link Widget} instances, following the composite pattern.
 * <p>
 * The children are stored in a list. The order of the list defines the front-to-back stacking order of the widgets
 * within the composite. The first widget in the list is at the back of the stacking order.
 * <p>
 * A widget cannot be added two times in a hierarchy.
 */
@SuppressWarnings("nls")
public class Composite extends Widget implements Iterable<Widget> {

	// list of widgets
	private final List<Widget> widgets;

	/**
	 * Creates a new composite.
	 * <p>
	 * Its bounds will be set to <code>0</code>.
	 */
	public Composite() {
		this.widgets = new ArrayList<>(0);
	}

	/**
	 * Disposes all composite children widgets. When a child is a composite, its children are disposed too.
	 */
	@Override
	public void dispose() {
		synchronized (this) {
			for (Widget widget : this) {
				try {
					widget.dispose();
				} catch (Throwable e) {
					// ignore error
				}
			}
			super.dispose();
		}
	}

	/**
	 * Adds the specified widget to the end of the list of children of this composite. If Widget <code>widget</code> is
	 * <code>null</code>, no exception is thrown and no action is performed.
	 *
	 * @param widget
	 *            the widget to add.
	 * @throws IllegalArgumentException
	 *             if the specified widget is already in a hierarchy (already contained in a composite).
	 */
	public void add(Widget widget) throws IllegalArgumentException {
		if (widget != null) {
			synchronized (this) {
				if (widget.getParent() != null) {
					throw new IllegalArgumentException("Widget is already available in the hierarchy.");
				}
				widget.setParent(this);
				this.widgets.add(widget);
			}
		}
	}

	/**
	 * Gets an iterator over the composite children widgets in proper sequence. When a child is a composite, the
	 * returned iterator iterates on its children.
	 * <p>
	 * The {@link Iterator#remove()} is not implemented (throws a {@link UnsupportedOperationException}).
	 *
	 * @return an iterator over the composite children widgets in proper sequence.
	 */
	@Override
	public Iterator<Widget> iterator() {
		return new WidgetIterator(this);
	}

	private static class WidgetIterator implements Iterator<Widget> {
		private final Deque<Iterator<Widget>> stack;

		WidgetIterator(Composite composite) {
			this.stack = new ArrayDeque<>();
			this.stack.push(composite.widgets.iterator());
		}

		@Override
		public boolean hasNext() {
			for (Iterator<Widget> iterator : this.stack) {
				if (iterator.hasNext()) {
					return true;
				}
			}
			return false;
		}

		@Override
		public Widget next() {
			Iterator<Widget> iterator = this.stack.peek();
			if (iterator.hasNext()) {
				Widget widget = iterator.next();
				if (widget instanceof Composite) {
					// push new iterator in the queue
					iterator = (((Composite) widget).widgets).iterator();
					this.stack.push(iterator);
				}
				return widget;
			}
			// remove iterator from queue
			this.stack.pop();
			// iterate on previous iterator
			return next();
		}
	}

}
