/*
 * Java
 *
 * Copyright 2017-2020 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.rcommand.synchronous;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeoutException;

import ej.rcommand.RemoteNotificationListener;
import ej.rcommand.impl.StreamRemoteConnection;

/**
 *
 */
public class RemoteCommandClient implements Runnable {

	private final Object connectionMonitor;
	private final StreamRemoteConnection connection;
	private final Map<Integer, Command<?>> pendingRequests;
	private final Map<String, RemoteNotificationListener> notificationListeners;
	private int nextId;
	private volatile boolean stopped;

	public RemoteCommandClient(StreamRemoteConnection connection) {
		this.connection = connection;
		this.connectionMonitor = new Object();
		this.pendingRequests = new HashMap<>();
		this.notificationListeners = new HashMap<>();
		new Thread(this, "RemoteCommandClient").start();
	}

	/**
	 * Registers the given notification listener.
	 *
	 * @param listener
	 *            the notification listener to register.
	 */
	public void registerNotificationListener(RemoteNotificationListener listener) {
		synchronized (this.connectionMonitor) {
			this.notificationListeners.put(listener.getName(), listener);
		}
	}

	@Override
	public void run() {
		while (!this.stopped) {
			try {
				String commandName = this.connection.readCommand();
				int commandId = this.connection.readInt();

				synchronized (this.connectionMonitor) {
					if (commandId == -1) {
						// handle notification
						RemoteNotificationListener listener = this.notificationListeners.get(commandName);
						if (listener != null) {
							listener.notificationReceived(this.connection);
						}
					} else {
						// handle command response
						Command<?> command = this.pendingRequests.remove(Integer.valueOf(commandId));
						if (command != null) {
							command.readResponse(this.connection);
							this.connectionMonitor.notifyAll();
						}
					}
					this.connection.skipParameters();
				}
			} catch (IllegalArgumentException e) {
				try {
					this.connection.skipParameters();
				} catch (IOException ioe) {
					// Ignored.
				}
			} catch (IOException e) {
				this.stopped = true;
			}
		}
	}

	/**
	 * Executes the command and waits for the result.
	 *
	 * @param command
	 *            the command to execute.
	 * @param timeout
	 *            the timeout in milliseconds.
	 * @return the result of the command.
	 * @throws IOException
	 *             if an I/O error occurs.
	 * @throws InterruptedException
	 * @throws TimeoutException
	 *             if the response of the command is not received before timeout.
	 */
	public <T> T execute(Command<T> command, long timeout)
			throws IOException, InterruptedException, TimeoutException {
		boolean infiniteWaiting = timeout == 0;

		synchronized (this.connectionMonitor) {
			int commandId = this.nextId++;
			this.pendingRequests.put(Integer.valueOf(commandId), command);
			this.connection.startCommand(command.getName());
			this.connection.sendInt(commandId);
			command.writeBody(this.connection);
			this.connection.flushCommand();

			long timeReference = System.currentTimeMillis();
			while (!command.isResponseReceived() && (timeout >= 0 || infiniteWaiting)) {
				if (!infiniteWaiting) {
					// Update remaining timeout.
					long newTimeReference = System.currentTimeMillis();
					timeout -= newTimeReference - timeReference;
					timeReference = newTimeReference;

					// To avoid exception or infinite waiting.
					if (timeout <= 0) {
						break;
					}
				}
				try {
					this.connectionMonitor.wait(timeout);
				} catch (InterruptedException e) {
					this.pendingRequests.remove(Integer.valueOf(commandId));
					throw e;
				}
			}

			this.pendingRequests.remove(Integer.valueOf(commandId));

			T response = command.getResponse();
			if (response == null) {
				throw new TimeoutException();
			}
			return response;
		}
	}

	/**
	 * Stops the client.
	 */
	public void stop() {
		this.stopped = false;
		this.connection.close();
	}
}
