/*
 * Copyright (c) 1996, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2017 IS2T. This file has been modified by IS2T S.A.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * (C) Copyright Taligent, Inc. 1996 - All Rights Reserved
 * (C) Copyright IBM Corp. 1996-1998 - All Rights Reserved
 *
 *   The original version of this source code and documentation is copyrighted
 * and owned by Taligent, Inc., a wholly-owned subsidiary of IBM. These
 * materials are provided under terms of a License Agreement between Taligent
 * and Sun. This technology is protected by multiple US and International
 * patents. This notice and attribution to Taligent may not be removed.
 *   Taligent is a registered trademark of Taligent, Inc.
 *
 */

package java.text;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;

/**
 * <code>SimpleDateFormat</code> is a concrete class for formatting and parsing dates in a locale-sensitive manner. It
 * allows for formatting (date &rarr; text), parsing (text &rarr; date), and normalization.
 *
 * <p>
 * <code>SimpleDateFormat</code> allows you to start by choosing any user-defined patterns for date-time formatting.
 * However, you are encouraged to create a date-time formatter with either <code>getTimeInstance</code>,
 * <code>getDateInstance</code>, or <code>getDateTimeInstance</code> in <code>DateFormat</code>. Each of these class
 * methods can return a date/time formatter initialized with a default format pattern. You may modify the format pattern
 * using the <code>applyPattern</code> methods as desired. For more information on using these methods, see
 * {@link DateFormat}.
 *
 * <h3>Date and Time Patterns</h3>
 * <p>
 * Date and time formats are specified by <em>date and time pattern</em> strings. Within date and time pattern strings,
 * unquoted letters from <code>'A'</code> to <code>'Z'</code> and from <code>'a'</code> to <code>'z'</code> are
 * interpreted as pattern letters representing the components of a date or time string. Text can be quoted using single
 * quotes (<code>'</code>) to avoid interpretation. <code>"''"</code> represents a single quote. All other characters
 * are not interpreted; they're simply copied into the output string during formatting or matched against the input
 * string during parsing.
 * <p>
 * The following pattern letters are defined (all other characters from <code>'A'</code> to <code>'Z'</code> and from
 * <code>'a'</code> to <code>'z'</code> are reserved): <blockquote>
 * <table border=0 cellspacing=3 cellpadding=0 summary=
 * "Chart shows pattern letters, date/time component, presentation, and examples.">
 * <tr style="background-color: rgb(204, 204, 255);">
 * <th align=left>Letter
 * <th align=left>Date or Time Component
 * <th align=left>Presentation
 * <th align=left>Examples
 * <tr>
 * <td><code>G</code>
 * <td>Era designator
 * <td><a href="#text">Text</a>
 * <td><code>AD</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>y</code>
 * <td>Year
 * <td><a href="#year">Year</a>
 * <td><code>1996</code>; <code>96</code>
 * <tr>
 * <td><code>Y</code>
 * <td>Week year
 * <td><a href="#year">Year</a>
 * <td><code>2009</code>; <code>09</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>M</code>
 * <td>Month in year
 * <td><a href="#month">Month</a>
 * <td><code>July</code>; <code>Jul</code>; <code>07</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>w</code>
 * <td>Week in year
 * <td><a href="#number">Number</a>
 * <td><code>27</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>W</code>
 * <td>Week in month
 * <td><a href="#number">Number</a>
 * <td><code>2</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>D</code>
 * <td>Day in year
 * <td><a href="#number">Number</a>
 * <td><code>189</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>d</code>
 * <td>Day in month
 * <td><a href="#number">Number</a>
 * <td><code>10</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>F</code>
 * <td>Day of week in month
 * <td><a href="#number">Number</a>
 * <td><code>2</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>E</code>
 * <td>Day name in week
 * <td><a href="#text">Text</a>
 * <td><code>Tuesday</code>; <code>Tue</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>u</code>
 * <td>Day number of week (1 = Monday, ..., 7 = Sunday)
 * <td><a href="#number">Number</a>
 * <td><code>1</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>a</code>
 * <td>Am/pm marker
 * <td><a href="#text">Text</a>
 * <td><code>PM</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>H</code>
 * <td>Hour in day (0-23)
 * <td><a href="#number">Number</a>
 * <td><code>0</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>k</code>
 * <td>Hour in day (1-24)
 * <td><a href="#number">Number</a>
 * <td><code>24</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>K</code>
 * <td>Hour in am/pm (0-11)
 * <td><a href="#number">Number</a>
 * <td><code>0</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>h</code>
 * <td>Hour in am/pm (1-12)
 * <td><a href="#number">Number</a>
 * <td><code>12</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>m</code>
 * <td>Minute in hour
 * <td><a href="#number">Number</a>
 * <td><code>30</code>
 * <tr bgcolor="#eeeeff">
 * <td><code>s</code>
 * <td>Second in minute
 * <td><a href="#number">Number</a>
 * <td><code>55</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>S</code>
 * <td>Millisecond
 * <td><a href="#number">Number</a>
 * <td><code>978</code>
 * </table>
 * </blockquote> Pattern letters are usually repeated, as their number determines the exact presentation:
 * <ul>
 * <li><strong><a name="text">Text:</a></strong> For formatting, if the number of pattern letters is 4 or more, the full
 * form is used; otherwise a short or abbreviated form is used if available. For parsing, both forms are accepted,
 * independent of the number of pattern letters.<br>
 * <br>
 * </li>
 * <li><strong><a name="number">Number:</a></strong> For formatting, the number of pattern letters is the minimum number
 * of digits, and shorter numbers are zero-padded to this amount. For parsing, the number of pattern letters is ignored
 * unless it's needed to separate two adjacent fields.<br>
 * <br>
 * </li>
 * <li><strong><a name="year">Year:</a></strong> If the formatter's {@link #getCalendar() Calendar} is the Gregorian
 * calendar, the following rules are applied.<br>
 * <ul>
 * <li>For formatting, if the number of pattern letters is 2, the year is truncated to 2 digits; otherwise it is
 * interpreted as a <a href="#number">number</a>.
 * <li>For parsing, if the number of pattern letters is more than 2, the year is interpreted literally, regardless of
 * the number of digits. So using the pattern "MM/dd/yyyy", "01/11/12" parses to Jan 11, 12 A.D.
 * <li>For parsing with the abbreviated year pattern ("y" or "yy"), <code>SimpleDateFormat</code> must interpret the
 * abbreviated year relative to some century. It does this by adjusting dates to be within 80 years before and 20 years
 * after the time the <code>SimpleDateFormat</code> instance is created. For example, using a pattern of "MM/dd/yy" and
 * a <code>SimpleDateFormat</code> instance created on Jan 1, 1997, the string "01/11/12" would be interpreted as Jan
 * 11, 2012 while the string "05/04/64" would be interpreted as May 4, 1964. During parsing, only strings consisting of
 * exactly two digits, as defined by {@link Character#isDigit(char)}, will be parsed into the default century. Any other
 * numeric string, such as a one digit string, a three or more digit string, or a two digit string that isn't all digits
 * (for example, "-1"), is interpreted literally. So "01/02/3" or "01/02/003" are parsed, using the same pattern, as Jan
 * 2, 3 AD. Likewise, "01/02/-3" is parsed as Jan 2, 4 BC.
 * </ul>
 * Otherwise system specific forms are applied. For both formatting and parsing, if the number of pattern letters is 4
 * or more, a calendar specific {@linkplain Calendar#LONG long form} is used. Otherwise, a calendar specific
 * {@linkplain Calendar#SHORT short or abbreviated form} is used.<br>
 * <br>
 * </li>
 * <li><strong><a name="month">Month:</a></strong> If the number of pattern letters is 3 or more, the month is
 * interpreted as <a href="#text">text</a>; otherwise, it is interpreted as a <a href="#number">number</a>.<br>
 * <br>
 * </li>
 * </ul>
 * <p>
 *
 * <h4>Examples</h4>
 *
 * The following examples show how date and time patterns are interpreted. <blockquote>
 * <table border=0 cellspacing=3 cellpadding=0 summary="Examples of date and time patterns interpreted">
 * <tr style="background-color: rgb(204, 204, 255);">
 * <th align=left>Date and Time Pattern
 * <th align=left>Result
 * <tr>
 * <td><code>"yyyy.MM.dd G 'at' HH:mm:ss z"</code>
 * <td><code>2001.07.04 AD at 12:08:56 PDT</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>"EEE, MMM d, ''yy"</code>
 * <td><code>Wed, Jul 4, '01</code>
 * <tr>
 * <td><code>"h:mm a"</code>
 * <td><code>12:08 PM</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>"hh 'o''clock' a, zzzz"</code>
 * <td><code>12 o'clock PM, Pacific Daylight Time</code>
 * <tr>
 * <td><code>"K:mm a, z"</code>
 * <td><code>0:08 PM, PDT</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>"yyyyy.MMMMM.dd GGG hh:mm aaa"</code>
 * <td><code>02001.July.04 AD 12:08 PM</code>
 * <tr>
 * <td><code>"EEE, d MMM yyyy HH:mm:ss Z"</code>
 * <td><code>Wed, 4 Jul 2001 12:08:56 -0700</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>"yyMMddHHmmssZ"</code>
 * <td><code>010704120856-0700</code>
 * <tr>
 * <td><code>"yyyy-MM-dd'T'HH:mm:ss.SSSZ"</code>
 * <td><code>2001-07-04T12:08:56.235-0700</code>
 * <tr style="background-color: rgb(238, 238, 255);">
 * <td><code>"yyyy-MM-dd'T'HH:mm:ss.SSSXXX"</code>
 * <td><code>2001-07-04T12:08:56.235-07:00</code>
 * <tr>
 * <td><code>"YYYY-'W'ww-u"</code>
 * <td><code>2001-W27-3</code>
 * </table>
 * </blockquote>
 *
 * <h4><a name="synchronization">Synchronization</a></h4>
 *
 * <p>
 * Date formats are not synchronized. It is recommended to create separate format instances for each thread. If multiple
 * threads access a format concurrently, it must be synchronized externally.
 *
 * @see <a href="https://docs.oracle.com/javase/tutorial/i18n/format/simpleDateFormat.html">Java Tutorial</a>
 * @see java.util.Calendar
 * @see DateFormat
 * @see DateFormatSymbols
 * @author Mark Davis, Chen-Lieh Huang, Alan Liu
 */
public class SimpleDateFormat extends DateFormat {
	private static final int HOUR_AT_MIDNIGHT = 24;
	private static final int HOUR_AT_MIDDAY = 12;

	private static final String EMPTY_STRING = "";
	/**
	 * Era designator.
	 */
	private static final char ERA_DESIGNATOR = 'G';
	/**
	 * Year.
	 */
	private static final char YEAR = 'y';
	/**
	 * Year.
	 */
	private static final char WEEK_YEAR = 'Y';
	/**
	 * Year.
	 */
	private static final char MONTH = 'M';
	/**
	 * Week in year.
	 */
	private static final char WEEK_OF_YEAR = 'w';
	/**
	 * Week in month.
	 */
	private static final char WEEK_OF_MONTH = 'W';
	/**
	 * Day in year.
	 */
	private static final char DAY_OF_YEAR = 'D';
	/**
	 * Day in month.
	 */
	private static final char DAY_OF_MONTH = 'd';
	/**
	 * Day of week in month.
	 */
	private static final char DAY_OF_WEEK_IN_MONTH = 'F';
	/**
	 * Day name in week.
	 */
	private static final char DAY_NAME_IN_WEEK = 'E';
	/**
	 * Day number of week (1 = Monday, ..., 7 = Sunday).
	 */
	private static final char DAY_NUMBER_OF_WEEK = 'u';
	/**
	 * Am/pm marker.
	 */
	private static final char AM_PM = 'a';
	/**
	 * Hour in day (0-23) .
	 */
	private static final char HOUR_OF_DAY_0 = 'H';
	/**
	 * Hour in day (1-24).
	 */
	private static final char HOUR_OF_DAY_1 = 'k';
	/**
	 * Hour in am/pm (1-12).
	 */
	private static final char HOUR_0 = 'K';
	/**
	 * Hour in am/pm (0-11).
	 */
	private static final char HOUR_1 = 'h';
	/**
	 * Minute in hour.
	 */
	private static final char MINUTE_IN_HOUR = 'm';
	/**
	 * Second in minute.
	 */
	private static final char SECOND_IN_MINUTE = 's';
	/**
	 * Millisecond.
	 */
	private static final char MILLISECOND = 'S';

	/**
	 * Marker for strings.
	 */
	private static final char STRING = '\'';

	private DateFormatSymbols symbols;
	private String pattern;
	private String[] compilePatterns;

	/**
	 * Constructs a <code>SimpleDateFormat</code> using the default pattern and date format symbols. <b>Note:</b> This
	 * constructor may not support all locales. For full coverage, use the factory methods in the {@link DateFormat}
	 * class.
	 */
	public SimpleDateFormat() {
		this(EMPTY_STRING, DateFormatSymbols.getInstance());
	}

	/**
	 * Constructs a <code>SimpleDateFormat</code> using the given pattern and the default date format symbols. For full
	 * coverage, use the factory methods in the {@link DateFormat} class.
	 * <p>
	 * This is equivalent to calling.
	 *
	 * @param pattern
	 *            the pattern describing the date and time format
	 * @exception NullPointerException
	 *                if the given pattern is null
	 * @exception IllegalArgumentException
	 *                if the given pattern is invalid
	 */
	public SimpleDateFormat(String pattern) {
		this(pattern, DateFormatSymbols.getInstance());
	}

	/**
	 * Constructs a <code>SimpleDateFormat</code> using the given pattern and date format symbols.
	 *
	 * @param pattern
	 *            the pattern describing the date and time format
	 * @param formatSymbols
	 *            the date format symbols to be used for formatting
	 * @exception NullPointerException
	 *                if the given pattern or formatSymbols is null
	 * @exception IllegalArgumentException
	 *                if the given pattern is invalid
	 */
	public SimpleDateFormat(String pattern, DateFormatSymbols formatSymbols) {
		if (pattern == null || formatSymbols == null) {
			throw new NullPointerException();
		}
		applyPattern(pattern);
		this.symbols = formatSymbols;
	}

	@Override
	public String format(Date date) {
		this.calendar.setTime(date);
		StringBuilder builder = new StringBuilder();
		for (String pattern : this.compilePatterns) {
			addPattern(builder, pattern);
		}
		return builder.toString();
	}

	private void addPattern(StringBuilder builder, String pattern) {
		char field = pattern.charAt(0);
		switch (field) {
		case ERA_DESIGNATOR:
			addERA(builder, pattern);
			break;
		case YEAR:
			addYear(builder, pattern);
			break;
		case WEEK_YEAR:
			addWeekYear(builder, pattern);
			break;
		case MONTH:
			addMonth(builder, pattern);
			break;
		case WEEK_OF_YEAR:
			addCalandarValue(Calendar.WEEK_OF_YEAR, builder, pattern);
			break;
		case WEEK_OF_MONTH:
			addCalandarValue(Calendar.WEEK_OF_MONTH, builder, pattern);
			break;
		case DAY_OF_YEAR:
			addCalandarValue(Calendar.DAY_OF_YEAR, builder, pattern);
			break;
		case DAY_OF_MONTH:
			addCalandarValue(Calendar.DAY_OF_MONTH, builder, pattern);
			break;
		case DAY_OF_WEEK_IN_MONTH:
			addCalandarValue(Calendar.DAY_OF_WEEK_IN_MONTH, builder, pattern);
			break;
		case DAY_NAME_IN_WEEK:
			addDay(builder, pattern);
			break;
		case DAY_NUMBER_OF_WEEK:
			addDayNumber(builder, pattern);
			break;
		case AM_PM:
			addAMPM(builder, pattern);
			break;
		case HOUR_OF_DAY_0:
			addHourOfDay(builder, pattern, false);
			break;
		case HOUR_OF_DAY_1:
			addHourOfDay(builder, pattern, true);
			break;
		case HOUR_0:
			addHour(builder, pattern, false);
			break;
		case HOUR_1:
			addHour(builder, pattern, true);
			break;
		case MINUTE_IN_HOUR:
			addCalandarValue(Calendar.MINUTE, builder, pattern);
			break;
		case SECOND_IN_MINUTE:
			addCalandarValue(Calendar.SECOND, builder, pattern);
			break;
		case MILLISECOND:
			addCalandarValue(Calendar.MILLISECOND, builder, pattern);
			break;
		case STRING:
			addString(builder, pattern);
			break;
		default:
			builder.append(pattern);
		}
	}

	private void addString(StringBuilder builder, String pattern) {
		if (pattern.length() <= 2) {
			builder.append(STRING);
		} else {
			pattern = pattern.substring(1, pattern.length() - 1);
			builder.append(pattern.replace(String.valueOf(STRING) + STRING, String.valueOf(STRING)));
		}
	}

	private void addHour(StringBuilder builder, String pattern, boolean startAtOne) {
		int hour = this.calendar.get(Calendar.HOUR);
		if (startAtOne && hour == 0) {
			hour = HOUR_AT_MIDDAY;
		}
		builder.append(formatNum(hour, pattern.length(), false));
	}

	private void addHourOfDay(StringBuilder builder, String pattern, boolean startAtOne) {
		int hour = this.calendar.get(Calendar.HOUR_OF_DAY);
		if (startAtOne && hour == 0) {
			hour = HOUR_AT_MIDNIGHT;
		}
		builder.append(formatNum(hour, pattern.length(), false));
	}

	private void addCalandarValue(int field, StringBuilder builder, String pattern) {
		int value = this.calendar.get(field);
		builder.append(formatNum(value, pattern.length(), false));
	}

	private void addAMPM(StringBuilder builder, String pattern) {
		int position;
		if (this.calendar.get(Calendar.AM_PM) == Calendar.AM) {
			position = 0;
		} else {
			position = 1;
		}
		builder.append(this.symbols.getAmPmString(position));
	}

	private void addDay(StringBuilder builder, String pattern) {
		int day = this.calendar.get(Calendar.DAY_OF_WEEK) - 1;
		String dayString;
		if (pattern.length() <= 3) {
			dayString = this.symbols.getShortWeekday(day);
		} else {
			dayString = this.symbols.getWeekday(day);
		}
		builder.append(dayString);
	}

	private void addDayNumber(StringBuilder builder, String pattern) {
		int day = this.calendar.get(Calendar.DAY_OF_WEEK) - 1;
		if (day == 0) { // SUNDAY
			day = 7;
		}
		builder.append(formatNum(day, pattern.length(), false));
	}

	private void addMonth(StringBuilder builder, String pattern) {
		int month = this.calendar.get(Calendar.MONTH);
		String monthString;
		if (pattern.length() == 3) {
			monthString = this.symbols.getShortMonth(month);
		} else if (pattern.length() > 3) {
			monthString = this.symbols.getMonth(month);
		} else {
			monthString = formatNum(month + 1, pattern.length(), false);
		}
		builder.append(monthString);
	}

	private void addYear(StringBuilder builder, String pattern) {
		int year = this.calendar.get(Calendar.YEAR);
		builder.append(formatNum(year, pattern.length(), pattern.length() == 2));
	}

	private void addWeekYear(StringBuilder builder, String pattern) {
		int year = this.calendar.get(Calendar.YEAR);
		int day = this.calendar.get(Calendar.DAY_OF_YEAR);
		int week = this.calendar.get(Calendar.WEEK_OF_YEAR);
		if (day < 7 && week > 1) {
			year--;
		} else if (day > 7 && week == 1) {
			year++;
		}
		builder.append(formatNum(year, pattern.length(), pattern.length() == 2));
	}

	private void addERA(StringBuilder builder, String pattern) {
		String era;
		if (this.calendar.get(Calendar.YEAR) < 0) {
			era = this.symbols.getEra(0);
		} else {
			era = this.symbols.getEra(1);
		}
		builder.append(era);
	}

	private static String formatNum(int value, int length, boolean truncate) {
		String number = Integer.toString(Math.abs(value));
		StringBuilder builder = new StringBuilder();
		int stringLength = number.length();
		if (stringLength > length && truncate) {
			builder.append(number.substring(stringLength - length, stringLength));
		} else {
			while (builder.length() + stringLength < length) {
				builder.append('0');
			}
			builder.append(number);
		}
		if (value < 0) {
			builder.insert(0, '-');
		}

		return builder.toString();
	}

	private static String getNextPattern(String pattern, int parsePosition) throws IllegalArgumentException {
		int size = pattern.length();
		if (parsePosition >= size) {
			return null;
		}
		int position = parsePosition + 1;
		char firstChar = pattern.charAt(parsePosition);

		if (firstChar == STRING) {
			boolean findEnd = false;
			while (true) {
				// We end the string without finding the end of the pattern.
				if (position == size) {
					if (!findEnd) {
						throw new IllegalArgumentException();
					} else {
						break;
					}
				}
				if (pattern.charAt(position) == STRING) {
					findEnd = !findEnd;
				} else if (findEnd) {
					break;
				}
				position++;
			}
		} else if (position < size && pattern.charAt(position) == firstChar) {
			while (position < size && pattern.charAt(position) == firstChar) {
				position++;
			}
		} else {
			return Character.toString(pattern.charAt(parsePosition));
		}
		return pattern.substring(parsePosition, position);
	}

	/**
	 * Returns a pattern string describing this date format.
	 *
	 * @return a pattern string describing this date format.
	 */
	public String toPattern() {
		return this.pattern;
	}

	/**
	 * Applies the given pattern string to this date format.
	 *
	 * @param pattern
	 *            the new date and time pattern for this date format
	 * @exception NullPointerException
	 *                if the given pattern is null
	 * @exception IllegalArgumentException
	 *                if the given pattern is invalid
	 */
	public void applyPattern(String pattern) {
		if (pattern == null) {
			throw new NullPointerException();
		}
		this.pattern = pattern;
		compilePatterns();
	}

	private void compilePatterns() throws IllegalArgumentException {
		List<String> strings = new ArrayList<>();
		int parsePosition = 0;
		String nextPattern = getNextPattern(this.pattern, parsePosition);
		while (nextPattern != null) {
			strings.add(nextPattern);
			parsePosition += nextPattern.length();
			nextPattern = getNextPattern(this.pattern, parsePosition);
		}
		String[] compilePatterns = new String[strings.size()];
		strings.toArray(compilePatterns);
		this.compilePatterns = compilePatterns;
	}

	/**
	 * Gets a copy of the date and time format symbols of this date format.
	 *
	 * @return the date and time format symbols of this date format
	 * @see #setDateFormatSymbols
	 */
	public DateFormatSymbols getDateFormatSymbols() {
		return this.symbols;
	}

	/**
	 * Sets the date and time format symbols of this date format.
	 *
	 * @param newFormatSymbols
	 *            the new date and time format symbols
	 * @exception NullPointerException
	 *                if the given newFormatSymbols is null
	 * @see #getDateFormatSymbols
	 */
	public void setDateFormatSymbols(DateFormatSymbols newFormatSymbols) {
		if (newFormatSymbols == null) {
			throw new NullPointerException();
		}
		this.symbols = newFormatSymbols;
	}

}
