/*
 * Copyright (c) 1996, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2017 IS2T. This file has been modified by IS2T S.A.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

/*
 * (C) Copyright Taligent, Inc. 1996 - All Rights Reserved
 * (C) Copyright IBM Corp. 1996 - All Rights Reserved
 *
 *   The original version of this source code and documentation is copyrighted
 * and owned by Taligent, Inc., a wholly-owned subsidiary of IBM. These
 * materials are provided under terms of a License Agreement between Taligent
 * and Sun. This technology is protected by multiple US and International
 * patents. This notice and attribution to Taligent may not be removed.
 *   Taligent is a registered trademark of Taligent, Inc.
 *
 */

package java.text;

import java.util.Calendar;
import java.util.Date;
import java.util.WeakHashMap;

/**
 * {@code DateFormat} is an abstract class for date/time formatting subclasses which formats and parses dates or time in
 * a language-independent manner. The date/time formatting subclass, such as {@link SimpleDateFormat}, allows for
 * formatting (i.e., date &rarr; text), parsing (text &rarr; date), and normalization. The date is represented as a
 * <code>Date</code> object or as the milliseconds since January 1, 1970, 00:00:00 GMT.
 *
 * <p>
 * {@code DateFormat} helps you to format and parse dates for any locale. Your code can be completely independent of the
 * locale conventions for months, days of the week, or even the calendar format: lunar vs. solar.
 *
 * <p>
 * To format a date for the current Locale, use one of the static factory methods: <blockquote>
 *
 * <pre>
 * {@code
 * myString = DateFormat.getDateInstance().format(myDate);
 * }
 * </pre>
 *
 * </blockquote>
 * <p>
 * If you are formatting multiple dates, it is more efficient to get the format and use it multiple times so that the
 * system doesn't have to fetch the information about the local language and country conventions multiple times.
 * <blockquote>
 *
 * <pre>
 * {
 * 	&#64;code
 * 	DateFormat df = DateFormat.getDateInstance();
 * 	for (int i = 0; i &lt; myDate.length; ++i) {
 * 		output.println(df.format(myDate[i]) + "; ");
 * 	}
 * }
 * </pre>
 *
 * </blockquote> <blockquote>
 *
 * <pre>
 * {
 * 	&#64;code
 * 	DateFormat df = DateFormat.getDateInstance(DateFormat.LONG, Locale.FRANCE);
 * }
 * </pre>
 *
 * </blockquote>
 * <p>
 * You can use a DateFormat to parse also. <blockquote>
 *
 * <pre>
 * {@code
 * myDate = df.parse(myString);
 * }
 * </pre>
 *
 * </blockquote>
 * <p>
 * Use {@code getDateInstance} to get the normal date format for that country. There are other static factory methods
 * available. Use {@code getTimeInstance} to get the time format for that country. Use {@code getDateTimeInstance} to
 * get a date and time format. You can pass in different options to these factory methods to control the length of the
 * result; from {@link #SHORT} to {@link #MEDIUM} to {@link #LONG} to {@link #FULL}. The exact result depends on the
 * locale, but generally:
 * <ul>
 * <li>{@link #SHORT} is completely numeric, such as {@code 12.13.52} or {@code 3:30pm}
 * <li>{@link #MEDIUM} is longer, such as {@code Jan 12, 1952}
 * <li>{@link #LONG} is longer, such as {@code January 12, 1952} or {@code 3:30:32pm}
 * <li>{@link #FULL} is pretty completely specified, such as {@code Tuesday, April 12, 1952 AD or 3:30:42pm PST}.
 * </ul>
 *
 * <p>
 * You can also set the time zone on the format if you wish. If you want even more control over the format or parsing,
 * (or want to give your users more control), you can try casting the {@code DateFormat} you get from the factory
 * methods to a {@link SimpleDateFormat}. This will work for the majority of countries; just remember to put it in a
 * {@code try} block in case you encounter an unusual one.
 *
 * <h3><a name="synchronization">Synchronization</a></h3>
 *
 * <p>
 * Date formats are not synchronized. It is recommended to create separate format instances for each thread. If multiple
 * threads access a format concurrently, it must be synchronized externally.
 *
 * @see SimpleDateFormat
 * @see java.util.Calendar
 * @see java.util.TimeZone
 * @author Mark Davis, Chen-Lieh Huang, Alan Liu
 */
public abstract class DateFormat {

	private static final int DATE_FLAG = 2;
	private static final int TIME_FLAG = 1;
	/**
	 * Constant for full style pattern.
	 */
	public static final int FULL = 0;
	/**
	 * Constant for long style pattern.
	 */
	public static final int LONG = 1;
	/**
	 * Constant for medium style pattern.
	 */
	public static final int MEDIUM = 2;
	/**
	 * Constant for short style pattern.
	 */
	public static final int SHORT = 3;
	/**
	 * Constant for default style pattern. Its value is MEDIUM.
	 */
	public static final int DEFAULT = MEDIUM;

	private static final WeakHashMap<String, DateFormat> FormaterPoll = new WeakHashMap<>();

	/**
	 * The {@link Calendar} instance used for calculating the date-time fields and the instant of time. This field is
	 * used for both formatting and parsing.
	 *
	 */
	protected Calendar calendar;

	/**
	 * Create a new date format.
	 */
	protected DateFormat() {
		this.calendar = Calendar.getInstance();
	}

	/**
	 * Formats a Date into a date/time string.
	 *
	 * @param date
	 *            the time value to be formatted into a time string.
	 * @return the formatted time string.
	 */
	public abstract String format(Date date);

	/**
	 * Gets the time formatter with the default formatting style.
	 *
	 * @return a time formatter.
	 */
	public static final DateFormat getTimeInstance() {
		return get(DEFAULT, 0, TIME_FLAG);
	}

	/**
	 * Gets the time formatter with the given formatting style.
	 *
	 * @param style
	 *            the given formatting style. For example, SHORT for "h:mm a".
	 * @return a time formatter.
	 */
	public static final DateFormat getTimeInstance(int style) {
		return get(style, 0, TIME_FLAG);
	}

	/**
	 * Gets the date formatter with the default formatting style.
	 *
	 * @return a date formatter.
	 */
	public static final DateFormat getDateInstance() {
		return get(0, DEFAULT, DATE_FLAG);
	}

	/**
	 * Gets the date formatter with the given formatting style.
	 *
	 * @param style
	 *            the given formatting style. For example, SHORT for "M/d/yy".
	 * @return a date formatter.
	 */
	public static final DateFormat getDateInstance(int style) {
		return get(0, style, DATE_FLAG);
	}

	/**
	 * Gets the date/time formatter with the default formatting style.
	 *
	 * @return a date/time formatter.
	 */
	public static final DateFormat getDateTimeInstance() {
		return get(DEFAULT, DEFAULT, TIME_FLAG | DATE_FLAG);
	}

	/**
	 * Gets the date/time formatter with the given date and time formatting styles.
	 *
	 * @param dateStyle
	 *            the given date formatting style. For example, SHORT for "M/d/yy".
	 * @param timeStyle
	 *            the given time formatting style. For example, SHORT for "h:mm a".
	 * @return a date/time formatter.
	 */
	public static final DateFormat getDateTimeInstance(int dateStyle, int timeStyle) {
		return get(timeStyle, dateStyle, TIME_FLAG | DATE_FLAG);
	}

	/**
	 * Get a default date/time formatter that uses the SHORT style for both the date and the time.
	 *
	 * @return a date/time formatter
	 */
	public static final DateFormat getInstance() {
		return getDateTimeInstance(SHORT, SHORT);
	}

	/**
	 * Set the calendar to be used by this date format. Initially, the default calendar.
	 *
	 * @param newCalendar
	 *            the new {@code Calendar} to be used by the date format
	 */
	public void setCalendar(Calendar newCalendar) {
		if (newCalendar == null) {
			throw new NullPointerException();
		}
		this.calendar = newCalendar;
	}

	/**
	 * Gets the calendar associated with this date/time formatter.
	 *
	 * @return the calendar associated with this date/time formatter.
	 */
	public Calendar getCalendar() {
		return this.calendar;
	}

	/**
	 * Creates a DateFormat with the given time and/or date style in the given locale.
	 *
	 * @param timeStyle
	 *            a value from 0 to 3 indicating the time format, ignored if flags is 2
	 * @param dateStyle
	 *            a value from 0 to 3 indicating the time format, ignored if flags is 1
	 * @param flags
	 *            either 1 for a time format, 2 for a date format, or 3 for a date/time format
	 * @param loc
	 *            the locale for the format
	 */
	private static DateFormat get(int timeStyle, int dateStyle, int flags) {
		if ((flags & TIME_FLAG) != 0) {
			if (timeStyle < FULL || timeStyle > SHORT) {
				throw new IllegalArgumentException("Illegal time style " + timeStyle); //$NON-NLS-1$
			}
		} else {
			timeStyle = -1;
		}
		if ((flags & DATE_FLAG) != 0) {
			if (dateStyle < FULL || dateStyle > SHORT) {
				throw new IllegalArgumentException("Illegal date style " + dateStyle); //$NON-NLS-1$
			}
		} else {
			dateStyle = -1;
		}
		// String time = getTimePattern(timeStyle);
		// String date = getDatePattern(dateStyle);
		StringBuilder builder = new StringBuilder();
		if (dateStyle != -1) {
			builder.append(getDatePattern(dateStyle));
			if (timeStyle != -1) {
				builder.append(" "); //$NON-NLS-1$
			}
		}
		if (timeStyle != -1) {
			builder.append(getTimePattern(timeStyle));
		}
		return getInstance(builder.toString());
	}

	private static DateFormat getInstance(String string) {
		DateFormat dateFormat = FormaterPoll.get(string);
		if (dateFormat == null) {
			dateFormat = new SimpleDateFormat(string);
			FormaterPoll.put(string, dateFormat);
		}
		return dateFormat;
	}

	private static String getDatePattern(int dateStyle) {
		String pattern;
		switch (dateStyle) {
		case SHORT:
			pattern = "MM/dd/yy"; //$NON-NLS-1$
			break;
		case MEDIUM:
			pattern = "MMM dd, y"; //$NON-NLS-1$
			break;
		case LONG:
			pattern = "MMMM dd, y"; //$NON-NLS-1$
			break;
		case FULL:
			pattern = "EEEE, MMMM dd, y"; //$NON-NLS-1$
			break;
		default:
			pattern = ""; //$NON-NLS-1$
			break;
		}
		return pattern;
	}

	private static String getTimePattern(int timeStyle) {
		String pattern;
		switch (timeStyle) {
		case SHORT:
		case MEDIUM:
			pattern = "h:mma"; //$NON-NLS-1$
			break;
		case LONG:
		case FULL:
			pattern = "h:mm:ssa"; //$NON-NLS-1$
			break;
		default:
			pattern = ""; //$NON-NLS-1$
			break;
		}
		return pattern;
	}
}