/*
 * Java
 *
 * Copyright 2016 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.tool.addon;

import java.io.File;

/**
 * A marker is an information set by a processor.
 */
public class Marker {

	/**
	 * A constant to indicate there is no line number information in a marker.
	 */
	public static final int NO_LINE_NUMBER = -1;

	/**
	 * An enumeration that represents possible marker levels.
	 */
	public enum MarkerLevel {
		/**
		 * Indicates that related marker is an information marker.
		 */
		Info,
		/**
		 * Indicates that related marker is a warning marker.
		 */
		Warning,
		/**
		 * Indicates that related marker is an error marker.
		 */
		Error
	}

	private final String _message; // NOSONAR
	private final Throwable _cause; // NOSONAR
	private final MarkerLevel _level; // NOSONAR

	private final int _line; // NOSONAR
	private final File _file; // NOSONAR

	/**
	 * Instantiates a new marker with given message and level.
	 *
	 * @param message
	 *            the message carried by the marker to create.
	 * @param level
	 *            the level of the marker to create.
	 */
	public Marker(String message, MarkerLevel level) {
		this(message, null, level);
	}

	/**
	 * Instantiates a new marker with given cause and level.
	 *
	 * @param cause
	 *            the cause of this marker.
	 * @param level
	 *            the level of the marker to create.
	 */
	public Marker(Throwable cause, MarkerLevel level) {
		this(null, cause, level);
	}

	/**
	 * Instantiates a new marker with given message, cause and level.
	 *
	 * @param message
	 *            the message carried by the marker to create.
	 * @param cause
	 *            the cause of this marker.
	 * @param level
	 *            the level of the marker to create.
	 */
	public Marker(String message, Throwable cause, MarkerLevel level) {
		this(message, null, NO_LINE_NUMBER, cause, level);
	}

	/**
	 * Instantiates a new marker with given message and level. Marker is attached to given file on given line number.
	 *
	 * @param message
	 *            the message carried by the marker to create.
	 * @param file
	 *            the file pointed by the marker to create.
	 * @param line
	 *            the line number of the file pointed the marker to create.
	 * @param level
	 *            the level of the marker to create.
	 */
	public Marker(String message, File file, int line, MarkerLevel level) {
		this(message, file, line, null, level);
	}

	/**
	 * Instantiates a new marker with given message, cause and level. Marker is attached to given file on given line
	 * number.
	 *
	 * @param message
	 *            the message carried by the marker to create.
	 * @param file
	 *            the file pointed by the marker to create.
	 * @param line
	 *            the line number of the file pointed the marker to create.
	 * @param cause
	 *            the cause of this marker.
	 * @param level
	 *            the level of the marker to create.
	 */
	public Marker(String message, File file, int line, Throwable cause, MarkerLevel level) {
		this._message = message;
		this._cause = cause;
		this._line = line;
		this._file = file;
		this._level = level;
	}

	/**
	 * Gets the message carried by this marker.
	 *
	 * @return the message carried by this marker, or {@code null} if this marker has no message.
	 */
	public String getMessage() {
		return this._message;
	}

	/**
	 * Gets the cause carried by this marker.
	 *
	 * @return the cause carried by this marker, or {@code null} if there is no cause to this marker.
	 */
	public Throwable getCause() {
		return this._cause;
	}

	/**
	 * Gets the file pointed by this marker.
	 *
	 * @return the file pointed by this marker, or {@code null} if no file is attached to this marker.
	 */
	public File getFile() {
		return this._file;
	}

	/**
	 * Gets the line number in the file pointed by this marker.
	 *
	 * @return the line number in the file pointed by this marker, or {@link #NO_LINE_NUMBER} if there is no line number
	 *         information.
	 */
	public int getLine() {
		return this._line;
	}

	/**
	 * Gets the level of this marker.
	 *
	 * @return the level of this marker.
	 */
	public MarkerLevel getLevel() {
		return this._level;
	}

}
