/*
 * Java
 *
 * Copyright 2019-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.library.iot.rcommand.bluetooth;

import java.io.IOException;
import java.util.concurrent.TimeoutException;

import ej.annotation.Nullable;
import ej.library.iot.rcommand.bluetooth.commands.AddServiceCommand;
import ej.library.iot.rcommand.bluetooth.commands.ConnectCommand;
import ej.library.iot.rcommand.bluetooth.commands.DisableCommand;
import ej.library.iot.rcommand.bluetooth.commands.DisconnectCommand;
import ej.library.iot.rcommand.bluetooth.commands.DiscoverServicesCommand;
import ej.library.iot.rcommand.bluetooth.commands.EnableCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendExecuteWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendNotificationCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPairRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPairResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPasskeyResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendPrepareWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendReadRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendReadResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteRequestCommand;
import ej.library.iot.rcommand.bluetooth.commands.SendWriteResponseCommand;
import ej.library.iot.rcommand.bluetooth.commands.StartAdvertisingCommand;
import ej.library.iot.rcommand.bluetooth.commands.StartScanningCommand;
import ej.library.iot.rcommand.bluetooth.commands.StopAdvertisingCommand;
import ej.library.iot.rcommand.bluetooth.commands.StopScanningCommand;
import ej.library.iot.rcommand.bluetooth.notifications.EventNotificationListener;
import ej.library.iot.rcommand.bluetooth.tools.BasicQueue;
import ej.rcommand.synchronous.Command;
import ej.rcommand.synchronous.RemoteCommandClient;

public class BluetoothHost {

	private static final int ENABLE_COMMAND_TIMEOUT = 5_000;

	private static final BluetoothHost INSTANCE = new BluetoothHost();

	private final BasicQueue<byte[]> eventQueue;

	private @Nullable RemoteCommandClient rcommandClient;
	private long commandTimeout;
	private boolean enabled;

	private BluetoothHost() {
		this.eventQueue = new BasicQueue<>();
		this.rcommandClient = null;
		this.commandTimeout = 0;
		this.enabled = false;
	}

	public static BluetoothHost getInstance() {
		return INSTANCE;
	}

	public void setup(RemoteCommandClient rcommandClient, long commandTimeout, int eventQueueSize) {
		rcommandClient.registerNotificationListener(new EventNotificationListener());

		this.eventQueue.resize(eventQueueSize);
		this.rcommandClient = rcommandClient;
		this.commandTimeout = commandTimeout;
	}

	public void pushEvent(byte[] event) {
		if (event.length > 0) {
			synchronized (this.eventQueue) {
				if (this.eventQueue.push(event)) {
					this.eventQueue.notifyAll();
				} else {
					System.out.println("WARNING: Bluetooth event discarded due to queue being full");
				}
			}
		}
	}

	public int waitEvent(byte[] buffer, int bufferLength) {
		synchronized (this.eventQueue) {
			while (this.enabled) {
				byte[] event = this.eventQueue.pop();
				if (event != null) {
					if (event.length > bufferLength) {
						System.out.println("WARNING: Bluetooth event discarded due to buffer being too small");
					} else {
						System.arraycopy(event, 0, buffer, 0, event.length);
						return event.length;
					}
				} else {
					try {
						this.eventQueue.wait();
					} catch (InterruptedException e) {
						e.printStackTrace();
					}
				}
			}
		}

		return 0;
	}

	public boolean enable() {
		this.enabled = sendBooleanCommand(new EnableCommand(), ENABLE_COMMAND_TIMEOUT);
		return this.enabled;
	}

	public void disable() {
		sendCommand(new DisableCommand(), this.commandTimeout);

		this.enabled = false;

		synchronized (this.eventQueue) {
			this.eventQueue.clear();
			this.eventQueue.notifyAll();
		}
	}

	public boolean startScanning(byte filterAction, byte filterType, byte[] filterData, int filterDataSize) {
		return sendBooleanCommand(new StartScanningCommand(filterAction, filterType, filterData, filterDataSize));
	}

	public boolean stopScanning() {
		return sendBooleanCommand(new StopScanningCommand());
	}

	public boolean startAdvertising(byte[] advertisementData, int advertisementDataSize) {
		return sendBooleanCommand(new StartAdvertisingCommand(advertisementData, advertisementDataSize));
	}

	public boolean stopAdvertising() {
		return sendBooleanCommand(new StopAdvertisingCommand());
	}

	public boolean connect(byte[] deviceAddress) {
		return sendBooleanCommand(new ConnectCommand(deviceAddress));
	}

	public boolean disconnect(short connHandle) {
		return sendBooleanCommand(new DisconnectCommand(connHandle));
	}

	public boolean sendPairRequest(short connHandle) {
		return sendBooleanCommand(new SendPairRequestCommand(connHandle));
	}

	public boolean sendPairResponse(short connHandle, boolean accept) {
		return sendBooleanCommand(new SendPairResponseCommand(connHandle, accept));
	}

	public boolean sendPasskeyResponse(short connHandle, boolean accept, int passkey) {
		return sendBooleanCommand(new SendPasskeyResponseCommand(connHandle, accept, passkey));
	}

	public boolean discoverServices(short connHandle, @Nullable byte[] uuid) {
		return sendBooleanCommand(new DiscoverServicesCommand(connHandle, uuid));
	}

	public boolean addService(byte[] service, short[] handles) {
		int numAttr = service[16] + service[17];
		int serviceSize = 18 + numAttr * 20;
		return sendBooleanCommand(new AddServiceCommand(service, serviceSize, handles));
	}

	public boolean sendReadRequest(short connHandle, short attributeHandle) {
		return sendBooleanCommand(new SendReadRequestCommand(connHandle, attributeHandle));
	}

	public boolean sendWriteRequest(short connHandle, short attributeHandle, byte[] value, int valueSize,
			boolean noResponse) {
		return sendBooleanCommand(
				new SendWriteRequestCommand(connHandle, attributeHandle, value, valueSize, noResponse));
	}

	public boolean sendReadResponse(short connHandle, short attributeHandle, byte status, byte[] value, int valueSize) {
		return sendBooleanCommand(new SendReadResponseCommand(connHandle, attributeHandle, status, value, valueSize));
	}

	public boolean sendWriteResponse(short connHandle, short attributeHandle, byte status) {
		return sendBooleanCommand(new SendWriteResponseCommand(connHandle, attributeHandle, status));
	}

	public boolean sendPrepareWriteResponse(short connHandle, short attributeHandle, byte status, byte[] value,
			int valueSize, int offset) {
		return sendBooleanCommand(
				new SendPrepareWriteResponseCommand(connHandle, attributeHandle, status, value, valueSize, offset));
	}

	public boolean sendExecuteWriteResponse(short connHandle, short attributeHandle, byte status) {
		return sendBooleanCommand(new SendExecuteWriteResponseCommand(connHandle, attributeHandle, status));
	}

	public boolean sendNotification(short connHandle, short attributeHandle, byte[] value, int valueSize,
			boolean confirm) {
		return sendBooleanCommand(new SendNotificationCommand(connHandle, attributeHandle, value, valueSize, confirm));
	}

	private @Nullable <T> T sendCommand(Command<T> command, long timeout) {
		RemoteCommandClient rcommandClient = this.rcommandClient;
		if (rcommandClient == null) {
			return null;
		}

		try {
			return rcommandClient.execute(command, timeout);
		} catch (IOException | InterruptedException | TimeoutException e) {
			System.out.println("WARNING: Bluetooth operation failed due to disconnection with Bluetooth Controller");
			return null;
		}
	}

	private boolean sendBooleanCommand(Command<Boolean> command, long timeout) {
		Boolean result = sendCommand(command, timeout);
		return (result != null && result.booleanValue());
	}

	private boolean sendBooleanCommand(Command<Boolean> command) {
		return sendBooleanCommand(command, this.commandTimeout);
	}
}
