/*
 * Copyright 2014-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket.frame;

import java.io.IOException;
import java.io.UnsupportedEncodingException;

import ej.bon.ByteArray;
import ej.util.message.Level;
import ej.websocket.CloseCodes;
import ej.websocket.Messages;
import ej.websocket.util.ArraysTools;
import ej.websocket.util.UTF8Validator;

/**
 * This is just a convenient class to encapsulate a byte array that contains a complete websocket frame. It provides
 * methods to get information and data from the frame.
 * <p>
 * The structure of a frame is described in RFC6455, section 5.2. Base Framing Protocol.
 *
 *
 *
 */
public final class RawFrame {

	private static final int BYTE_MAX_VALUE = 0x7F;

	// RFC 6455 defines a list of valid opcodes in the section 5.2. Base Framing Protocol. See page 29.
	/**
	 * Opcode to denote a continuation frame.
	 */
	public static final byte OPCODE_CONTINUATION_FRAME = 0x00;

	/**
	 * Opcode to denote a text frame.
	 */
	public static final byte OPCODE_TEXT_FRAME = 0x01;

	/**
	 * Opcode to denote a binary frame.
	 */
	public static final byte OPCODE_BINARY_FRAME = 0x02;

	// 0x3-0x7 are reserved for further non-control frames

	/**
	 * Opcode to denote a connection close.
	 */
	public static final byte OPCODE_CONNECTION_CLOSE = 0x08;

	/**
	 * Opcode to denote a ping.
	 */
	public static final byte OPCODE_PING = 0x09;

	/**
	 * Opcode to denote a pong.
	 */
	public static final byte OPCODE_PONG = 0x0A;

	// "0xB-0xF are reserved for further control frames"

	// The bytes that form the entire frame
	private byte[] bytes;

	/**
	 * <code>bytes</code> will be copied internally so that it is not tied to the newly created instance and can then be
	 * reused.
	 *
	 * @param bytes
	 *            the data used to create the frame
	 * @throws IllegalArgumentException
	 *             if <code>bytes</code> is null
	 */
	public RawFrame(byte[] bytes) throws IllegalArgumentException {

		if (bytes == null) {
			throw new NullPointerException();
		}

		this.bytes = bytes.clone();
		// TODO future versions may validate the frame
	}

	/**
	 * Note: this is not a copy of the internal representation. Handle it with care so as not to alter the frame.
	 *
	 * @return the byte array of the frame
	 */
	public byte[] getBytes() {
		return this.bytes;
	}

	/**
	 * FIN is a bit. It is the LSB of the returned byte.
	 *
	 * @return FIN
	 */
	public byte getFIN() {
		return (byte) ((this.bytes[0] & 0x80) >> 7);
	}

	/**
	 * RSV are 3 bits. They are the LSB of the returned byte.
	 *
	 * @return RSV[1..3]
	 */
	public byte getRSV() {
		return (byte) ((this.bytes[0] & 0x70) >> 4);
	}

	/**
	 * Opcode is 4 bits. They are the LSB of the returned byte.
	 *
	 * @return opcode
	 */
	public byte getOpcode() {
		return (byte) (this.bytes[0] & 0x0F);
	}

	/**
	 * Test the opcode of this frame against possible opcodes provided by RFC6455.
	 *
	 * @return 'true' if 'code' is a valid opcode, 'false' otherwise
	 */
	public boolean hasValidOpcode() {
		byte code = this.getOpcode();
		return (code == OPCODE_CONTINUATION_FRAME) || (code == OPCODE_TEXT_FRAME) || (code == OPCODE_BINARY_FRAME)
				|| (code == OPCODE_CONNECTION_CLOSE) || (code == OPCODE_PING) || (code == OPCODE_PONG);
	}

	/**
	 * MASK is a bit. It is the LSB of the returned byte.
	 *
	 * @return FIN
	 */
	public byte getMASK() {
		return (byte) ((this.bytes[1] & 0x80) >> 7);
	}

	/**
	 * Tell whether this frame is a control frame or not.
	 *
	 * @return 'true' is this is a control frame, 'false' otherwise
	 */
	public boolean isControlFrame() {
		// Section 5.5: "Control frames are identified by opcodes where the most significant bit of the opcode is 1."
		byte opcode = getOpcode();
		return (opcode & 0x08) != 0;
	}

	/**
	 * A convenient method to test whether the frame is masked or not.
	 *
	 * @return true if MASK bit is set to 1; false otherwise
	 */
	public boolean isMasked() {
		return getMASK() == 1;
	}

	/**
	 * Tell whether is valid or not, according to the RFC6455.
	 *
	 * @return 'true' if the frame is valid, 'false' otherwise
	 * @throws IOException
	 *             if the frame does not respect the RFC6455
	 * @throws UnsupportedEncodingException
	 *             if the frame does not validate the UTF8 format
	 */
	public boolean isValid() throws IOException {
		// Opcode must be valid
		if (!hasValidOpcode()) {
			throw new IOException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.OPCODE_NOT_MANAGED));
			// In this, we must _Fail the WebSocket Connection_ (see page 29)
		}

		// Text frame payload must be UTF-8 valid
		if (getOpcode() == OPCODE_TEXT_FRAME && getLength() != 0) {
			long payloadLength = getPayloadLength();
			if (payloadLength > Integer.MAX_VALUE) {
				throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
						Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
			}
			// Check reason
			if (!UTF8Validator.isValid(this.bytes, getPayloadOffset(), (int) payloadLength)) {
				throw new UnsupportedEncodingException(
						Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.INVALID_UTF8));
			}
		}

		// Control frames have special constraints (see section 5.5):
		if (isControlFrame()) {
			isControlValid();
		}

		// So far, so good
		return true;
	}

	private void isControlValid() throws IOException, UnsupportedEncodingException {
		// - cannot be fragmented
		if (getFIN() != 1) {
			throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.NOT_FIN));
		}

		// - payload length cannot be greater than 125
		if (getLength() > 125) {
			throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
					Messages.INVALID_LENGTH, getLength()));
		}

		// Close control frames have even more specific constraints
		if (getOpcode() == OPCODE_CONNECTION_CLOSE) {
			// Payload load length must be 0 (no code, no reason) and greater-or-equal than 2 (code and optionnal
			// reason)

			if (getLength() == 1) {
				// This is not allowed, need at least two bytes to create the close code
				throw new IOException(
						Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.INVALID_FRAME));
			}

			if (getLength() >= 2) {
				// Extract code
				byte[] payload = getPayload(true);
				int msb = payload[0] & 0xFF;
				int lsb = payload[1] & 0xFF;
				int code = (msb << 8) + lsb;

				// Check code
				if (!CloseCodes.canBeUsedInCloseFrame(code)) {
					throw new IOException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
							Messages.INVALID_CLOSE_CODE, code));
				}

				// Check optional close reason
				if (!UTF8Validator.isValid(payload, 2, payload.length - 2)) {
					throw new UnsupportedEncodingException(
							Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.INVALID_UTF8));
				}
			}
		}
	}

	// Length is a bit more complicated, since it can be extended to 16 or even 64 bits
	/**
	 * A frame has a field called payload length. This is the value returned here. This may not be the actual length
	 * since a frame can have an additional field called "extended payload length".
	 *
	 * @return the payload length
	 */
	public byte getLength() {
		return (byte) (this.bytes[1] & BYTE_MAX_VALUE);
	}

	/**
	 * Tell whether this frame has an extended length field.
	 *
	 * @return 'true' if it has an extended length field.
	 */
	public boolean hasExtendedLength() {
		return getLength() == BYTE_MAX_VALUE - 1;
	}

	/**
	 * Tell whether this frame has an extended length field.
	 *
	 * @return 'true' if it has an extended length field.
	 */
	public boolean hasVeryExtendedLength() {
		return getLength() == BYTE_MAX_VALUE;
	}

	/**
	 * Get the extended length of the frame.
	 *
	 * @return the extended length is such a field is present in this frame; 0 otherwise
	 */
	public long getExtendedLength() {
		if (hasExtendedLength()) {
			int msb = this.bytes[2] & 0xFF;
			int lsb = this.bytes[3] & 0xFF;
			return (msb << 8) + lsb;
		} else if (hasVeryExtendedLength()) {
			return ByteArray.readLong(this.bytes, 2, ByteArray.BIG_ENDIAN);
		} else {
			return 0;
		}
	}

	public void setLength(long length) {
		long previousLength = getPayloadLength();
		if (previousLength >= BYTE_MAX_VALUE && length < BYTE_MAX_VALUE) {
			this.bytes = ArraysTools.removeRange(this.bytes, 2, 2);
		} else if (previousLength < BYTE_MAX_VALUE && length >= BYTE_MAX_VALUE) {
			this.bytes = ArraysTools.insertRange(this.bytes, 2, 2);
		}
		this.bytes[1] |= BYTE_MAX_VALUE;
		this.bytes[1] = (byte) (Math.min(length, BYTE_MAX_VALUE - 1l) & BYTE_MAX_VALUE);

		if (length >= BYTE_MAX_VALUE) {
			this.bytes[2] = (byte) ((length >> 8) & 0xFF);
			this.bytes[3] = (byte) (length & 0xFF);
		}
	}

	/**
	 * Masking key is an optional field of the websocket protocol. If present, it is 4 byte long.
	 *
	 * @return null if the frame is not masked; otherwise, the masking key in a fresh new 4 byte array
	 */
	public byte[] getMaskingKey() {
		if (isMasked()) {
			byte[] maskingKey = new byte[4];
			int offset = hasExtendedLength() ? 4 : 2;
			System.arraycopy(this.bytes, offset, maskingKey, 0, 4);
			return maskingKey;
		} else {
			return null;
		}
	}

	/**
	 * The payload (= 'real' data) of this frame.
	 *
	 * @param unmaskIfNeeded
	 *            if 'true' and MASK is 1 then it returns the unmasked payload; otherwise, the payload is returned
	 *            directly
	 * @return the payload in a fresh new array
	 */
	public byte[] getPayload(boolean unmaskIfNeeded) {
		if (isMasked() && unmaskIfNeeded) {
			// Frame unmasking is not implemented.
			throw new UnsupportedOperationException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.UNSUPPORTED_MASKED));
			// TODO future versions will implement payload unmasking

		} else {
			// Position of payload is not fixed because of optional masking key and extended length
			int offset = getPayloadOffset();
			long payloadLength = getPayloadLength();
			if (payloadLength > Integer.MAX_VALUE) {
				throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
						Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
			}

			byte[] payload = new byte[(int) payloadLength];
			System.arraycopy(this.bytes, offset, payload, 0, (int) payloadLength);
			return payload;
		}
	}

	public String getPayloadAsString(boolean unmaskIfNeeded) {
		if (isMasked() && unmaskIfNeeded) {
			// Frame unmasking is not implemented.
			throw new UnsupportedOperationException(
					Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY, Messages.UNSUPPORTED_MASKED));
			// TODO future versions will implement payload unmasking

		} else {
			// Position of payload is not fixed because of optional masking key and extended length
			int offset = getPayloadOffset();
			long payloadLength = getPayloadLength();
			if (payloadLength > Integer.MAX_VALUE) {
				throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
						Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
			}

			return new String(this.bytes, offset, (int) payloadLength);
		}
	}

	/**
	 * Gets the payload length.
	 *
	 * @return the payload length.
	 */
	public long getPayloadLength() {
		long payloadLength = getLength();

		if (hasExtendedLength() || hasVeryExtendedLength()) {
			payloadLength = getExtendedLength();
		}

		return payloadLength;
	}

	private int getPayloadOffset() {
		int offset = 2;

		if (hasExtendedLength()) {
			offset += 2;
		} else if (hasVeryExtendedLength()) {
			offset += 8;
		}

		if (isMasked()) {
			offset += 4;
		}
		return offset;
	}

	/**
	 * Appends the payload of a frame into the current frame.
	 *
	 * @param f
	 *            the frame to gets the payload from.
	 */
	public synchronized void append(RawFrame f) {
		int currentOffset = getPayloadOffset();
		long currentPayloadLength = getPayloadLength();
		byte[] fBytes = f.getBytes();
		long currentLength = currentOffset + currentPayloadLength;
		long fPayloadLength = f.getPayloadLength();
		long length = fPayloadLength + currentPayloadLength;
		if (currentLength > Integer.MAX_VALUE || fPayloadLength > Integer.MAX_VALUE) {
			throw new UnsupportedOperationException(Messages.BUILDER.buildMessage(Level.SEVERE, Messages.CATEGORY,
					Messages.UNSUPPORTED_VERY_EXTENDED_LENGTH));
		}
		this.bytes = ArraysTools.add(this.bytes, 0, (int) currentLength, fBytes, f.getPayloadOffset(),
				(int) fPayloadLength);
		setLength(length);
	}
}
