/*
 * Copyright 2019-2025 MicroEJ Corp.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.websocket;

import ej.util.message.MessageBuilder;
import ej.util.message.MessageLogger;
import ej.util.message.basic.BasicMessageBuilder;
import ej.util.message.basic.BasicMessageLogger;
import ej.util.message.basic.OffMessageLogger;

/**
 * Gather error messages.
 */
public final class Messages {

	// ****************//
	// Error messages. //
	// ****************//
	/**
	 * Unknown error.
	 */
	public static final int ERROR_UNKNOWN = -255;

	/**
	 * We received an OP code that is not managed.
	 */
	public static final int OPCODE_NOT_MANAGED = -1;

	/**
	 * We received an UTF8 frame that has an unvalid payload.
	 */
	public static final int INVALID_UTF8 = -2;

	/**
	 * This implementation does not support masked frame.
	 */
	public static final int UNSUPPORTED_MASKED = -3;
	/**
	 * This implementation does not support very extended length.
	 */
	public static final int UNSUPPORTED_VERY_EXTENDED_LENGTH = -4;

	/**
	 * This is not a fin frame.
	 */
	public static final int NOT_FIN = -5;
	/**
	 * Payload length cannot be greater than 125.
	 */
	public static final int INVALID_LENGTH = -6;

	/**
	 * The close code is not valid.
	 */
	public static final int INVALID_CLOSE_CODE = -7;

	/**
	 * An invalid sequence of fragmented frame arrived.
	 */
	public static final int INVALID_SEQUENCE = -8;

	// ****************//
	// Info messages. //
	// ****************//
	/**
	 * A timeout has been set.
	 */
	public static final int TIMEOUT_START = 0;
	/**
	 * The timeout has been closed normally.
	 */
	public static final int TIMEOUT_NORMAL_CLOSE = 1;
	/**
	 * The timeout has been reached.
	 */
	public static final int TIMEOUT = 2;

	/**
	 * The timeout has been interrupted.
	 */
	public static final int TIMEOUT_INTERRUPTED = 3;

	/**
	 * The receiver is started.
	 */
	public static final int RECEIVER_START = 4;

	/**
	 * One frame has been received.
	 */
	public static final int RECEIVER_FRAME_RECEIVED = 5;

	/**
	 * If we get here and no exception occurred, it means that the connection has been closed. We don't have to call
	 * onClose() on the endpoint since the connection is already closed (and consequently onClose() has been called
	 * already.
	 */
	public static final int RECEIVER_TERMINATED = 6;

	/**
	 * This case happens when the OnTimeOutCloser closes the underlying TCP connection. Hence, the websocket state is
	 * CLOSED and the TCP connection is closed so an IOException occurs in getFrame() because its calls read() on an
	 * invalid input stream. There is nothing to do since both websocket and TCP socket (and associated streams) are
	 * closed.
	 */
	public static final int RECEIVER_TIMEOUT = 7;

	/**
	 * We can receive valid packets if the connection is not in the OPEN state. Example: someone ask to close the
	 * connection (user can do this because it chooses to do so, validate() can this because we receive an invalid
	 * frame) so the connection moves to the CLOSING state. At this moment, we can receive a ping frame. handle() will
	 * try to response with a pong frame and an IllegalstateException will be raised.
	 */
	public static final int VALID_PACKET_CONNECTION_CLOSED = 8;

	/**
	 * We received an invalid frame.
	 */
	public static final int INVALID_FRAME = 9;

	/**
	 * We received a text frame.
	 */
	public static final int RECEIVED_TEXT_FRAME = 10;

	/**
	 * We received a binary frame.
	 */
	public static final int RECEIVED_BINARY_FRAME = 11;
	/**
	 * We received a close frame.
	 */
	public static final int RECEIVED_CLOSE_FRAME = 12;
	/**
	 * We received a ping frame.
	 */
	public static final int RECEIVED_PING_FRAME = 13;
	/**
	 * We received a pong frame.
	 */
	public static final int RECEIVED_PONG_FRAME = 14;

	/**
	 * A close has been request.
	 */
	public static final int CLOSE = 15;

	/**
	 * The websocket is already closed.
	 */
	public static final int ALREADY_CLOSE = 16;

	/**
	 * A send of text has been requested.
	 */
	public static final int SEND_TEXT = 17;
	/**
	 * A send of binary has been requested.
	 */
	public static final int SEND_BINARY = 18;
	/**
	 * A ping has been requested.
	 */
	public static final int PING = 19;
	/**
	 * A pong has been requested.
	 */
	public static final int PONG = 20;

	/**
	 * Respond to a closing handshake.
	 */
	public static final int RESPOND_CLOSING = 21;

	/**
	 * Close TCP connection.
	 */
	public static final int CLOSE_TCP = 22;

	/**
	 * Receive some data.
	 */
	public static final int RECEIVE = 23;

	/**
	 * Sending the opening handshake.
	 */
	public static final int OPENING_HANDSHAKE = 24;

	/**
	 * Sending some data.
	 */
	public static final int SENDING = 25;

	/**
	 * The status code is invalid.
	 */
	public static final int INVALID_STATUS = 26;

	/**
	 * Invalid upgrade field.
	 */
	public static final int INVALID_UPGRADE = 27;

	/**
	 * Invalid connection field.
	 */
	public static final int INVALID_CONNECTION = 28;

	/**
	 * Missing Sec-WebSocket-Accept header.
	 */
	public static final int INVALID_SEC_WEBSOCKET_ACCEPT = 29;
	/**
	 * Category message errors.
	 */
	public static final String CATEGORY = Messages.class.getPackage().getName();

	/**
	 * The message builder.
	 */
	public static final MessageBuilder BUILDER;

	/**
	 * The message logger.
	 */
	public static MessageLogger LOGGER;


	static {
		BUILDER = BasicMessageBuilder.INSTANCE;
		log(false);
	}

	private Messages() {
		// Forbid instantiation
	}

	/**
	 *
	 * @param log
	 */
	public static void log(boolean log) {
		if(log) {
			LOGGER = BasicMessageLogger.INSTANCE;
		} else {
			LOGGER = OffMessageLogger.INSTANCE;
		}

	}

}
