/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import static org.junit.Assert.assertEquals;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.bon.Constants;
import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.BlendMode;
import ej.microvg.Matrix;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.Direction;

/**
 * Tests kerning.
 */
@SuppressWarnings("nls")
public class TestFontKerning {

	private static final float DELTA = 0.02f;
	private static final int PADDING = 1;
	private static final String STRING_KERNING = "IHIHJ";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		TestUtilities.clearScreen();
	}

	/**
	 * Test drawing a string with kerning.
	 *
	 */
	@Test
	public void testKerningDrawString2bytes() {
		testStringKerning(STRING_KERNING, 1);
	}

	private void testStringKerning(String string, int nbChar) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();
		float fontSize = 64;
		float charWidth = TestUtilities.measureCharWidth(string.substring(0, nbChar), font, fontSize);
		int color = Colors.WHITE;

		int x = display.getWidth() / 4;
		int y = display.getHeight() / 2;

		float glyphX = x;
		float glyphWidth = charWidth * 0.6f;
		float glyphY = y + fontSize * 0.4f;
		float glyphHeight = fontSize * 0.6f;

		g.setColor(color);
		VectorGraphicsPainter.drawString(g, string, font, fontSize, x, y);
		display.flush();

		// Check first I (square)
		TestUtilities.checkArea("kerning first I character", color, (int) glyphX, (int) glyphY, (int) glyphWidth,
				(int) glyphHeight, PADDING);
		TestUtilities.checkPeripheralArea("kerning first I character", TestUtilities.BACKGROUND_COLOR, (int) glyphX, (int) glyphY,
				(int) glyphWidth, (int) glyphHeight, 10, 3);

		// Check first H
		glyphX = x + charWidth;
		TestUtilities.checkPeripheralArea("kerning first H character", TestUtilities.BACKGROUND_COLOR, (int) glyphX, y, (int) fontSize,
				(int) fontSize, 10, 3);

		// Check second I (square) X= X+2*charWidth-kerning with kerning = glyphWidth
		glyphX = x + 2 * charWidth - glyphWidth;
		TestUtilities.checkArea("kerning second I character", color, (int) glyphX, (int) glyphY, (int) glyphWidth,
				(int) glyphHeight, PADDING);

		// Check J square X= X0+4*charWidth-kerning with kerning = glyphWidth
		glyphX = x + 4 * charWidth - glyphWidth;
		TestUtilities.checkArea("kerning last J character", color, (int) glyphX, (int) glyphY, (int) glyphWidth,
				(int) glyphHeight, PADDING);
	}

	/**
	 * Test drawing a string with kerning.
	 *
	 */
	@Test
	public void testKerningDrawStringOnCircle2bytes() {
		testStringOnCircleKerning(STRING_KERNING, 1);
	}

	/**
	 * Test drawing a string with kerning.
	 */
	private void testStringOnCircleKerning(String string, int nbChar) {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		VectorFont font = TestUtilities.getTestFont();

		int color = Colors.WHITE;
		float fontSize = 50;
		float radius = 150;
		float topCircleRadius = radius + font.getBaselinePosition(fontSize);
		float charWidth = TestUtilities.measureCharWidth(string.substring(0, nbChar), font, fontSize);

		int x = display.getWidth() / 2;
		int y = display.getHeight() / 2;
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);

		float glyphWidth = charWidth * 0.6f;

		if (Constants.getBoolean(TestUtilities.DEBUG_CONSTANT)) {
			TestUtilities.drawStringOnCircleGrid(font, fontSize, radius, x, y, Direction.CLOCKWISE);
		}

		g.setColor(color);
		VectorGraphicsPainter.drawStringOnCircle(g, string, font, fontSize, matrix, radius, Direction.CLOCKWISE,
				GraphicsContext.OPAQUE, BlendMode.SRC_OVER, 0);
		display.flush();

		// compute glyph position
		// angle offset is negative in clockwise direction
		float charAngle = -charWidth / (radius);
		float glyphAngle = -glyphWidth / (topCircleRadius - fontSize);
		float angleToGlyphCenter = glyphAngle / 2;

		// Check first I glyph
		float currentAngle = angleToGlyphCenter;
		int targetX = (int) (x + radius * Math.cos(currentAngle));
		int targetY = (int) (y - radius * Math.sin(currentAngle));

		TestUtilities.check("kerning first I character", targetX, targetY, g, color);

		// Check second I glyph
		currentAngle = 2 * charAngle - angleToGlyphCenter;
		targetX = (int) (x + radius * Math.cos(currentAngle));
		targetY = (int) (y - radius * Math.sin(currentAngle));

		TestUtilities.check("kerning second I character", targetX, targetY, g, color);

		// Check second J glyph
		currentAngle = 4 * charAngle - angleToGlyphCenter;
		targetX = (int) (x + radius * Math.cos(currentAngle));
		targetY = (int) (y - radius * Math.sin(currentAngle));

		TestUtilities.check("kerning last J character", targetX, targetY, g, color);
	}

	/**
	 * Test string width with kerning with characters represented on 2 bytes.
	 */
	@Test
	public void testKerningStringWidth2bytes() {
		testKerningStringWidth(STRING_KERNING);
	}

	private void testKerningStringWidth(String string) {
		float fontSize = 100;
		float kerning = fontSize * 0.6f;
		VectorFont font = TestUtilities.getTestFont();
		float stringWidth = TestUtilities.measureCharWidth(string, font, fontSize);
		assertEquals(fontSize * 5 - kerning, stringWidth, DELTA);
	}
}
