/*
 * Copyright (c) 1995, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2015-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package java.net;

import java.io.IOException;

import ej.annotation.Nullable;

/**
 * The multicast datagram socket class is useful for sending and receiving IP multicast packets. A MulticastSocket is a
 * (UDP) DatagramSocket, with additional capabilities for joining "groups" of other multicast hosts on the internet.
 * <P>
 * A multicast group is specified by a class D IP address and by a standard UDP port number. Class D IP addresses are in
 * the range <CODE>224.0.0.0</CODE> to <CODE>239.255.255.255</CODE>, inclusive. The address 224.0.0.0 is reserved and
 * should not be used.
 * <P>
 * One would join a multicast group by first creating a MulticastSocket with the desired port, then invoking the
 * <CODE>joinGroup(InetAddress groupAddr)</CODE> method:
 *
 * <PRE>
 * // join a Multicast group and send the group salutations
 * ...
 * String msg = "Hello";
 * InetAddress group = InetAddress.getByName("228.5.6.7");
 * MulticastSocket s = new MulticastSocket(6789);
 * s.joinGroup(group);
 * DatagramPacket hi = new DatagramPacket(msg.getBytes(), msg.length(),
 *                             group, 6789);
 * s.send(hi);
 * // get their responses!
 * byte[] buf = new byte[1000];
 * DatagramPacket recv = new DatagramPacket(buf, buf.length);
 * s.receive(recv);
 * ...
 * // OK, I'm done talking - leave the group...
 * s.leaveGroup(group);
 * </PRE>
 *
 * When one sends a message to a multicast group, <B>all</B> subscribing recipients to that host and port receive the
 * message (within the time-to-live range of the packet, see below). The socket needn't be a member of the multicast
 * group to send messages to it.
 * <P>
 * When a socket subscribes to a multicast group/port, it receives datagrams sent by other hosts to the group/port, as
 * do all other members of the group and port. A socket relinquishes membership in a group by the leaveGroup(InetAddress
 * addr) method. <B> Multiple MulticastSocket's</B> may subscribe to a multicast group and port concurrently, and they
 * will all receive group datagrams.
 * <P>
 * Currently applets are not allowed to use multicast sockets.
 *
 * @author Pavani Diwanji
 * @since JDK1.1
 */
public class MulticastSocket extends DatagramSocket {

	/**
	 * Create a multicast socket.
	 *
	 * <p>
	 * If there is a security manager, its {@code checkListen} method is first called with 0 as its argument to ensure
	 * the operation is allowed. This could result in a SecurityException.
	 *
	 * @exception IOException
	 *                if an I/O exception occurs while creating the MulticastSocket
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkListen} method doesn't allow the operation.
	 */
	public MulticastSocket() throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Create a multicast socket and bind it to a specific port.
	 *
	 * <p>
	 * If there is a security manager, its {@code checkListen} method is first called with the {@code port} argument as
	 * its argument to ensure the operation is allowed. This could result in a SecurityException.
	 *
	 * @param port
	 *            port to use
	 * @exception IOException
	 *                if an I/O exception occurs while creating the MulticastSocket
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkListen} method doesn't allow the operation.
	 */
	public MulticastSocket(int port) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Create a MulticastSocket bound to the specified socket address.
	 * <p>
	 * Or, if the address is {@code null}, create an unbound socket.
	 *
	 * <p>
	 * If there is a security manager, its {@code checkListen} method is first called with the SocketAddress port as its
	 * argument to ensure the operation is allowed. This could result in a SecurityException.
	 *
	 * @param bindaddr
	 *            Socket address to bind to, or {@code null} for an unbound socket.
	 * @exception IOException
	 *                if an I/O exception occurs while creating the MulticastSocket
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkListen} method doesn't allow the operation.
	 *
	 * @since 1.4
	 */
	public MulticastSocket(@Nullable SocketAddress bindaddr) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Set the default time-to-live for multicast packets sent out on this {@code MulticastSocket} in order to control
	 * the scope of the multicasts.
	 *
	 * <P>
	 * The ttl <B>must</B> be in the range {@code  0 <= ttl <=
	 * 255} or an {@code IllegalArgumentException} will be thrown. Multicast packets sent with a TTL of {@code 0} are
	 * not transmitted on the network but may be delivered locally.
	 *
	 * @param ttl
	 *            the time-to-live
	 *
	 * @throws IOException
	 *             if an I/O exception occurs while setting the default time-to-live value
	 *
	 * @see #getTimeToLive()
	 */
	public void setTimeToLive(int ttl) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Get the default time-to-live for multicast packets sent out on the socket.
	 *
	 * @exception IOException
	 *                if an I/O exception occurs while getting the default time-to-live value
	 * @return the default time-to-live value
	 * @see #setTimeToLive(int)
	 */
	public int getTimeToLive() throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Joins a multicast group. Its behavior may be affected by {@code setInterface} or {@code setNetworkInterface}.
	 *
	 * <p>
	 * If there is a security manager, this method first calls its {@code checkMulticast} method with the
	 * {@code mcastaddr} argument as its argument.
	 *
	 * @param mcastaddr
	 *            is the multicast address to join
	 *
	 * @exception IOException
	 *                if there is an error joining or when the address is not a multicast address.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkMulticast} method doesn't allow the join.
	 */
	public void joinGroup(InetAddress mcastaddr) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Leave a multicast group. Its behavior may be affected by {@code setInterface} or {@code setNetworkInterface}.
	 *
	 * <p>
	 * If there is a security manager, this method first calls its {@code checkMulticast} method with the
	 * {@code mcastaddr} argument as its argument.
	 *
	 * @param mcastaddr
	 *            is the multicast address to leave
	 * @exception IOException
	 *                if there is an error leaving or when the address is not a multicast address.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkMulticast} method doesn't allow the operation.
	 */
	public void leaveGroup(InetAddress mcastaddr) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Joins the specified multicast group at the specified interface.
	 *
	 * <p>
	 * If there is a security manager, this method first calls its {@code checkMulticast} method with the
	 * {@code mcastaddr} argument as its argument.
	 *
	 * @param mcastaddr
	 *            is the multicast address to join
	 * @param netIf
	 *            specifies the local interface to receive multicast datagram packets, or <i>null</i> to defer to the
	 *            interface set by {@link MulticastSocket#setInterface(InetAddress)} or
	 *            {@link MulticastSocket#setNetworkInterface(NetworkInterface)}
	 *
	 * @exception IOException
	 *                if there is an error joining or when the address is not a multicast address.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkMulticast} method doesn't allow the join.
	 * @throws IllegalArgumentException
	 *             if mcastaddr is null or is a SocketAddress subclass not supported by this socket
	 * @since 1.4
	 */
	public void joinGroup(SocketAddress mcastaddr, @Nullable NetworkInterface netIf) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Leave a multicast group on a specified local interface.
	 *
	 * <p>
	 * If there is a security manager, this method first calls its {@code checkMulticast} method with the
	 * {@code mcastaddr} argument as its argument.
	 *
	 * @param mcastaddr
	 *            is the multicast address to leave
	 * @param netIf
	 *            specifies the local interface or <i>null</i> to defer to the interface set by
	 *            {@link MulticastSocket#setInterface(InetAddress)} or
	 *            {@link MulticastSocket#setNetworkInterface(NetworkInterface)}
	 * @exception IOException
	 *                if there is an error leaving or when the address is not a multicast address.
	 * @exception SecurityException
	 *                if a security manager exists and its {@code checkMulticast} method doesn't allow the operation.
	 * @throws IllegalArgumentException
	 *             if mcastaddr is null or is a SocketAddress subclass not supported by this socket
	 * @since 1.4
	 */
	public void leaveGroup(SocketAddress mcastaddr, @Nullable NetworkInterface netIf) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Set the multicast network interface used by methods whose behavior would be affected by the value of the network
	 * interface. Useful for multihomed hosts.
	 *
	 * @param inf
	 *            the InetAddress
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @see #getInterface()
	 */
	public void setInterface(InetAddress inf) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Retrieve the address of the network interface used for multicast packets.
	 *
	 * @return An {@code InetAddress} representing the address of the network interface used for multicast packets.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 *
	 * @see #setInterface(java.net.InetAddress)
	 */
	public InetAddress getInterface() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Specify the network interface for outgoing multicast datagrams sent on this socket.
	 *
	 * @param netIf
	 *            the interface
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @see #getNetworkInterface()
	 * @since 1.4
	 */
	public void setNetworkInterface(NetworkInterface netIf) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Get the multicast network interface set.
	 *
	 * @exception SocketException
	 *                if there is an error in the underlying protocol, such as a TCP error.
	 * @return the multicast {@code NetworkInterface} currently set
	 * @see #setNetworkInterface(NetworkInterface)
	 * @since 1.4
	 */
	public NetworkInterface getNetworkInterface() throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Disable/Enable local loopback of multicast datagrams The option is used by the platform's networking code as a
	 * hint for setting whether multicast data will be looped back to the local socket.
	 *
	 * <p>
	 * Because this option is a hint, applications that want to verify what loopback mode is set to should call
	 * {@link #getLoopbackMode()}
	 *
	 * @param disable
	 *            {@code true} to disable the LoopbackMode
	 * @throws SocketException
	 *             if an error occurs while setting the value
	 * @since 1.4
	 * @see #getLoopbackMode
	 */
	public void setLoopbackMode(boolean disable) throws SocketException {
		throw new RuntimeException();
	}

	/**
	 * Get the setting for local loopback of multicast datagrams.
	 *
	 * @throws SocketException
	 *             if an error occurs while getting the value
	 * @return true if the LoopbackMode has been disabled
	 * @since 1.4
	 * @see #setLoopbackMode
	 */
	public boolean getLoopbackMode() throws SocketException {
		throw new RuntimeException();
	}
}
