/*
 * Copyright (c) 1998, 2013, Oracle and/or its affiliates. All rights reserved.
 * Copyright 2015-2022 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation. The Copyright holder(s) designates
 * this particular file as subject to the "Classpath" exception as provided
 * by the Copyright holder(s) in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact the Copyright holder(s) if you need additional information
 * or have any questions.
 * (if Oracle: 500 Oracle Parkway, Redwood Shores, CA 94065 USA or visit www.oracle.com)
 */

package java.security.cert;

import java.io.InputStream;

/**
 * This class defines the functionality of a certificate factory, which is used to generate certificate, certification
 * path ({@code CertPath}) and certificate revocation list (CRL) objects from their encodings.
 *
 * <p>
 * For encodings consisting of multiple certificates, use {@code generateCertificates} when you want to parse a
 * collection of possibly unrelated certificates. Otherwise, use {@code generateCertPath} when you want to generate a
 * {@code CertPath} (a certificate chain) and subsequently validate it with a {@code CertPathValidator}.
 *
 * <p>
 * A certificate factory for X.509 must return certificates that are an instance of
 * {@code java.security.cert.X509Certificate}, and CRLs that are an instance of {@code java.security.cert.X509CRL}.
 *
 * <p>
 * The following example reads a file with Base64 encoded certificates, which are each bounded at the beginning by
 * -----BEGIN CERTIFICATE-----, and bounded at the end by -----END CERTIFICATE-----. We convert the
 * {@code FileInputStream} (which does not support {@code mark} and {@code reset}) to a {@code BufferedInputStream}
 * (which supports those methods), so that each call to {@code generateCertificate} consumes only one certificate, and
 * the read position of the input stream is positioned to the next certificate in the file:
 *
 * <pre>
 * FileInputStream fis = new FileInputStream(filename);
 * BufferedInputStream bis = new BufferedInputStream(fis);
 *
 * CertificateFactory cf = CertificateFactory.getInstance("X.509");
 *
 * while (bis.available() &#62; 0) {
 * 	Certificate cert = cf.generateCertificate(bis);
 * 	System.out.println(cert.toString());
 * }
 * </pre>
 *
 * <p>
 * The following example parses a PKCS#7-formatted certificate reply stored in a file and extracts all the certificates
 * from it:
 *
 * <pre>
 * FileInputStream fis = new FileInputStream(filename);
 * CertificateFactory cf = CertificateFactory.getInstance("X.509");
 * Collection c = cf.generateCertificates(fis);
 * Iterator i = c.iterator();
 * while (i.hasNext()) {
 * 	Certificate cert = (Certificate) i.next();
 * 	System.out.println(cert);
 * }
 * </pre>
 *
 * <p>
 * Every implementation of the Java platform is required to support the following standard {@code CertificateFactory}
 * type:
 * <ul>
 * <li>{@code X.509}</li>
 * </ul>
 * and the following standard {@code CertPath} encodings:
 * <ul>
 * <li>{@code PKCS7}</li>
 * <li>{@code PkiPath}</li>
 * </ul>
 * The type and encodings are described in the
 * <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#CertificateFactory"> CertificateFactory
 * section</a> and the <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#CertPathEncodings"> CertPath
 * Encodings section</a> of the Java Cryptography Architecture Standard Algorithm Name Documentation. Consult the
 * release documentation for your implementation to see if any other types or encodings are supported.
 *
 * @author Hemma Prafullchandra
 * @author Jan Luehe
 * @author Sean Mullan
 *
 * @see Certificate
 * @see X509Certificate
 *
 * @since 1.2
 */

public class CertificateFactory {

	/**
	 * Returns a certificate factory object that implements the specified certificate type.
	 *
	 * <p>
	 * This method traverses the list of registered security Providers, starting with the most preferred Provider. A new
	 * CertificateFactory object encapsulating the CertificateFactorySpi implementation from the first Provider that
	 * supports the specified type is returned.
	 *
	 *
	 * @param type
	 *            the name of the requested certificate type. See the CertificateFactory section in the
	 *            <a href= "{@docRoot}/../technotes/guides/security/StandardNames.html#CertificateFactory"> Java
	 *            Cryptography Architecture Standard Algorithm Name Documentation</a> for information about standard
	 *            certificate types.
	 *
	 * @return a certificate factory object for the specified type.
	 *
	 * @exception CertificateException
	 *                if no Provider supports a CertificateFactorySpi implementation for the specified type.
	 *
	 */
	public static final CertificateFactory getInstance(String type) throws CertificateException {
		throw new RuntimeException();
	}

	/**
	 * Returns the name of the certificate type associated with this certificate factory.
	 *
	 * @return the name of the certificate type associated with this certificate factory.
	 */
	public final String getType() {
		throw new RuntimeException();
	}

	/**
	 * Generates a certificate object and initializes it with the data read from the input stream {@code inStream}.
	 *
	 * <p>
	 * In order to take advantage of the specialized certificate format supported by this certificate factory, the
	 * returned certificate object can be typecast to the corresponding certificate class. For example, if this
	 * certificate factory implements X.509 certificates, the returned certificate object can be typecast to the
	 * {@code X509Certificate} class.
	 *
	 * <p>
	 * In the case of a certificate factory for X.509 certificates, the certificate provided in {@code inStream} must be
	 * DER-encoded and may be supplied in binary or printable (Base64) encoding. If the certificate is provided in
	 * Base64 encoding, it must be bounded at the beginning by -----BEGIN CERTIFICATE-----, and must be bounded at the
	 * end by -----END CERTIFICATE-----.
	 *
	 * <p>
	 * Note that if the given input stream does not support {@link java.io.InputStream#mark(int) mark} and
	 * {@link java.io.InputStream#reset() reset}, this method will consume the entire input stream. Otherwise, each call
	 * to this method consumes one certificate and the read position of the input stream is positioned to the next
	 * available byte after the inherent end-of-certificate marker. If the data in the input stream does not contain an
	 * inherent end-of-certificate marker (other than EOF) and there is trailing data after the certificate is parsed, a
	 * {@code CertificateException} is thrown.
	 *
	 * @param inStream
	 *            an input stream with the certificate data.
	 *
	 * @return a certificate object initialized with the data from the input stream.
	 *
	 * @exception CertificateException
	 *                on parsing errors.
	 */
	public final Certificate generateCertificate(InputStream inStream) throws CertificateException {
		throw new RuntimeException();
	}

}
