/*
 * Copyright 2022-2023 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.widget.debug;

import ej.microui.display.Colors;
import ej.microui.display.Font;
import ej.mwt.style.background.Background;
import ej.mwt.style.background.RoundedBackground;
import ej.mwt.style.dimension.Dimension;
import ej.mwt.style.dimension.FixedDimension;
import ej.mwt.style.dimension.OptimalDimension;
import ej.mwt.style.dimension.RelativeDimension;
import ej.mwt.style.outline.Outline;
import ej.mwt.style.outline.border.RoundedBorder;
import ej.mwt.util.Alignment;

/* package */ class Stringifier {

	private Stringifier() {
		// Forbid instantiation
	}

	public static String toString(Dimension dimension) {
		String string;
		if (dimension instanceof FixedDimension) {
			string = toString((FixedDimension) dimension);
		} else if (dimension instanceof OptimalDimension) {
			string = toString((OptimalDimension) dimension);
		} else if (dimension instanceof RelativeDimension) {
			string = toString((RelativeDimension) dimension);
		} else {
			string = getClassName(dimension);
		}
		return string;
	}

	/**
	 * Serializes the given color.
	 * <p>
	 * Attempts to detect a named color. Otherwise, the hexa-decimal format {@code #RRGGBB} is used.
	 *
	 * @param color
	 *            the color to serialize.
	 * @return the serialized color.
	 * @see Colors named colors
	 */
	public static String colorToString(int color) {
		switch (color) {
		// @formatter:off
		case Colors.BLACK: return "black";
		case Colors.BLUE: return "blue";
		case Colors.CYAN: return "cyan";
		case Colors.GRAY: return "gray";
		case Colors.GREEN: return "green";
		case Colors.LIME: return "lime";
		case Colors.MAGENTA: return "magenta";
		case Colors.MAROON: return "maroon";
		case Colors.NAVY: return "navy";
		case Colors.OLIVE: return "olive";
		case Colors.PURPLE: return "purple";
		case Colors.RED: return "red";
		case Colors.SILVER: return "silver";
		case Colors.TEAL: return "teal";
		case Colors.WHITE: return "white";
		case Colors.YELLOW: return "yellow";
		// @formatter:on
		default:
			String hex = Integer.toHexString(color);
			char[] chars = new char[7];
			chars[0] = '#';
			for (int i = 1, n = hex.length(), len = chars.length; i < len; i++) {
				int j = i + n - len;
				chars[i] = j < 0 ? '0' : hex.charAt(j);
			}
			return new String(chars);
		}
	}

	public static String toString(Outline border) {
		String string;
		if (border instanceof RoundedBorder) {
			string = toString((RoundedBorder) border);
		} else {
			string = getClassName(border);
		}
		return string;
	}

	public static String toString(Background background) {
		String string;
		if (background instanceof RoundedBackground) {
			string = toString((RoundedBackground) background);
		} else {
			string = getClassName(background);
		}
		return string;
	}

	public static String toString(Font font) {
		String string = getClassName(font);
		String descriptor = font.getDescriptor();
		if (!descriptor.isEmpty()) {
			string += "[" + descriptor + "]";
		}
		return string;
	}

	public static String alignmentToString(int alignment) {
		switch (alignment) {
		// @formatter:off
		case Alignment.BOTTOM: return "bottom";
		case Alignment.HCENTER: return "hcenter";
		case Alignment.LEFT: return "left";
		case Alignment.RIGHT: return "right";
		case Alignment.TOP: return "top";
		case Alignment.VCENTER: return "vcenter";
		default: throw new IllegalArgumentException();
		// @formatter:on
		}
	}

	private static String getClassName(Object object) {
		return getClassName(object.getClass());
	}

	private static String getClassName(Class<?> clazz) {
		String simpleName = clazz.getSimpleName();
		int length = simpleName.length();
		if (simpleName.charAt(length - 1) == '@' && simpleName.charAt(0) != '@') {
			simpleName = simpleName.substring(0, length - 1);
		}
		return simpleName;
	}

	private static String toString(RoundedBackground roundedBackground) {
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(getClassName(RoundedBackground.class));
		stringBuilder.append("[color=").append(colorToString(roundedBackground.getColor()));
		stringBuilder.append(", radius=").append(roundedBackground.getCornerRadius());
		stringBuilder.append(", thickness=").append(roundedBackground.getBorderThickness());
		stringBuilder.append(']');
		return stringBuilder.toString();
	}

	private static String toString(RoundedBorder roundedBorder) {
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(getClassName(RoundedBorder.class));
		stringBuilder.append("[color=").append(colorToString(roundedBorder.getColor()));
		stringBuilder.append(", radius=").append(roundedBorder.getCornerRadius());
		stringBuilder.append(", thickness=").append(roundedBorder.getThickness());
		stringBuilder.append(']');
		return stringBuilder.toString();
	}

	public static String toString(FixedDimension dimension) {
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(getClassName(FixedDimension.class));
		stringBuilder.append("[width=").append(dimension.getWidth());
		stringBuilder.append(", height=").append(dimension.getHeight());
		stringBuilder.append(']');
		return stringBuilder.toString();
	}

	public static String toString(OptimalDimension dimension) {
		String type;
		if (dimension == OptimalDimension.OPTIMAL_DIMENSION_X) {
			type = "X";
		} else if (dimension == OptimalDimension.OPTIMAL_DIMENSION_Y) {
			type = "Y";
		} else if (dimension == OptimalDimension.OPTIMAL_DIMENSION_XY) {
			type = "XY";
		} else {
			type = "unknown";
		}
		return getClassName(OptimalDimension.class) + "[" + type + "]";
	}

	public static String toString(RelativeDimension dimension) {
		StringBuilder stringBuilder = new StringBuilder();
		stringBuilder.append(getClassName(FixedDimension.class));
		stringBuilder.append("[widthRatio=").append(dimension.getWidthRatio());
		stringBuilder.append(", heightRatio=").append(dimension.getHeightRatio());
		stringBuilder.append(']');
		return stringBuilder.toString();
	}

}
