/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.module.SandboxedModuleHelper;
import ej.service.ServiceFactory;

import java.net.Socket;

/**
 * Action performed when writing on a socket.
 */
public class WriteSocket extends AbstractSocket {

	private final int nbBytes;

	/**
	 * Creates a write socket action.
	 *
	 * @param socket
	 * 		the socket of the write action.
	 * @param nbBytes
	 * 		the number of written bytes.
	 */
	public WriteSocket(Socket socket, int nbBytes) {
		super(socket);
		this.nbBytes = nbBytes;
	}

	/**
	 * Gets the number of bytes to be written.
	 *
	 * @return the number of bytes to be written.
	 */
	public int getNbBytes() {
		return this.nbBytes;
	}

	/**
	 * Called when the execution of the action is about to start.
	 *
	 * This hook is used to check if the action is allowed to be performed.
	 *
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public void onStart() throws SecurityException {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getNetworkResourceController().onStart(currentModule, this);
	}

	/**
	 * Called when the execution of the action is finished.
	 *
	 * This hook is used to update the current resources state of the module that performs the action.
	 *
	 * @param endWithSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public void onEnd(boolean endWithSuccess) {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getNetworkResourceController().onEnd(currentModule, this, endWithSuccess);
	}
}
