/*
 * Copyright 2015-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Widget;

/**
 * Represents a selector.
 */
public interface Selector {

	/**
	 * Checks whether or not this selector applies to the given widget.
	 *
	 * @param widget
	 *            the widget to test.
	 * @return <code>true</code> if this selectors applies to the given widget, <code>false</code> otherwise.
	 */
	boolean appliesToWidget(Widget widget);

	/**
	 * Returns the specificity of this selector.
	 * <p>
	 * The specificity allows to determine in which order the rules of a stylesheet should be applied.
	 * <p>
	 * A specificity is composed of four numbers (defined by CSS2 specification):
	 * <ul>
	 * <li>count 1 if the declaration is from is a 'style' attribute rather than a rule with a selector, 0 otherwise (=
	 * a)</li>
	 * <li>count the number of ID attributes in the selector (= b)</li>
	 * <li>count the number of other attributes and pseudo-classes in the selector (= c)</li>
	 * <li>count the number of widget names and pseudo-widgets in the selector (= d)</li>
	 * </ul>
	 * Concatenating the four numbers a-b-c-d (in a number system with a large base) gives the specificity. In this
	 * implementation, each number is stored on a byte in an integer. {@link SelectorHelper} provides a method to
	 * compute the specificity of a selector.
	 *
	 * @return the specificity of this selector.
	 */
	int getSpecificity();

	// define this method to avoid embedding the equals() method of every class (M0081MEJA-1240)
	@Override
	boolean equals(@Nullable Object obj);
}
