/*
 * Java
 *
 * Copyright 2016-2024 MicroEJ Corp. This file has been modified and/or created by MicroEJ Corp.
 */
package ej.junit;

import java.io.ByteArrayOutputStream;
import java.io.PrintStream;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Assume;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

/**
 * Base class used by generated wrappers. It is where the test execution happens. Implements the {@link TestWrapper}
 * interface so it can be called by another application.
 */
public abstract class AbstractTestWrapper implements TestWrapper {

	protected final Class<?> testClass;
	protected final String testClassName;

	/**
	 * Creates a wrapper instance for the specified test class.
	 * <p>
	 * Queries and store the class name, so clients of the wrappers can look them up by name if necessary.
	 *
	 * @param testClass
	 *            the test class to instantiate.
	 */
	public AbstractTestWrapper(Class<?> testClass) {
		this.testClass = testClass;
		this.testClassName = testClass.getName();
	}

	protected boolean failed = false;
	protected String testMethod;
	protected long testMethodStart;
	protected Object test;

	/* package */int runs;
	/* package */int failures;
	/* package */int errors;
	protected TestListener testListener;

	/**
	 * Called by generated wrappers to report an error.
	 * <p>
	 * If a listener has been passed to {@link #run(TestListener)}, then the
	 * {@link TestListener#onError(String, String, String)} is called.
	 *
	 * @param error
	 *            the {@link Throwable} that caused the error.
	 * @see #run(TestListener)
	 */
	protected void reportError(Throwable error) {
		this.errors++;
		if (this.testListener != null) {
			ByteArrayOutputStream stackStream = new ByteArrayOutputStream();
			error.printStackTrace(new PrintStream(stackStream));
			this.testListener.onError(error.getMessage(), error.getClass().getName(), stackStream.toString());
		}
	}

	/**
	 * Called by generated wrappers to report a failure.
	 * <p>
	 * If a listener has been passed to {@link #run(TestListener)}, then the
	 * {@link TestListener#onFailure(String, String, String)} is called.
	 *
	 * @param error
	 *            the {@link AssertionError} that caused the failure.
	 * @see #run(TestListener)
	 */
	protected void reportFailure(AssertionError error) {
		this.failures++;
		if (this.testListener != null) {
			ByteArrayOutputStream stackStream = new ByteArrayOutputStream();
			error.printStackTrace(new PrintStream(stackStream));
			this.testListener.onFailure(error.getMessage(), error.getClass().getName(), stackStream.toString());
		}
	}

	/**
	 * Called by generated wrappers to report that a test was skipped.
	 * <p>
	 * If a listener has been passed to {@link #run(TestListener)}, then the {@link TestListener#onSkip(String)} is
	 * called.
	 *
	 * @param message
	 *            the message associated with the test skip
	 * @see #run(TestListener)
	 */
	protected void reportSkip(String message) {
		if (this.testListener != null) {
			this.testListener.onSkip(message);
		}
	}

	protected void reportTestEnd() {
		if (this.testListener != null) {
			double duration = (System.currentTimeMillis() - this.testMethodStart) / 1000.0;
			this.testListener.onCaseEnd(this.testMethod, duration);
		}
	}

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link BeforeClass}
	 *
	 * @throws Exception
	 *             if one of the {@link BeforeClass} methods throws one.
	 */
	abstract protected void runBeforeClassMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link AfterClass}
	 *
	 * @throws Exception
	 *             if one of the {@link AfterClass} methods throws one.
	 */
	abstract protected void runAfterClassMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link Before}
	 *
	 * @throws Exception
	 *             if one of the {@link Before} methods throws one.
	 */
	abstract protected void runBeforeMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of methods tagged with {@link After}
	 *
	 * @throws Exception
	 *             if one of the {@link BeforeClass} methods throws one.
	 */
	abstract protected void runAfterMethods() throws Exception;

	/**
	 * To be implemented by generated subclasses, triggers the execution of test methods tagged with {@link Test}
	 */
	abstract protected void runTestMethods();

	/* package */ boolean wrapperRunBeforeClassMethod() {
		try {
			this.testMethod = null;
			runBeforeClassMethods();
			return true;
		} catch (Throwable t) {
			reportError(t);
		}
		return false;
	}

	/* package */void wrapperRunAfterClassMethod() {
		try {
			this.testMethod = null;
			runAfterClassMethods();
		} catch (Throwable t) {
			reportError(t);
		}
	}

	/**
	 * Called by generated subclasses to trigger test initialization, by instantiating the test class and calling the
	 * {@link Before} methods.
	 *
	 * @param testMethodName
	 *            the name of the test method to initialize.
	 * @return true if the operation succeeded and the test can continue, false otherwise.
	 */
	protected final boolean testInitialize(String testMethodName) {
		try {
			this.runs++;
			if (this.testListener != null) {
				this.testListener.testInstance(this.testClass.getName());
			}
			this.test = this.testClass.newInstance();
			this.testMethod = testMethodName;

			runBeforeMethods();

			this.testMethodStart = System.currentTimeMillis();

			// test will actually start right after that
			if (this.testListener != null) {
				this.testListener.onCaseStart(testMethodName, this.testClassName, "FILE", 0); //$NON-NLS-1$
			}
		} catch (Throwable t) {
			reportError(t);
			return false;
		}

		final String property = "tests." + this.testClass.getSimpleName() + "." + testMethodName + ".skip"; //$NON-NLS-1$ //$NON-NLS-2$ //$NON-NLS-3$
		Assume.assumeFalse(property + " skips this test", Boolean.getBoolean(property)); //$NON-NLS-1$

		return true;
	}

	/**
	 * Called by generated subclasses to trigger test finalization. Reports the test end and calls the After methods.
	 */
	protected final void testFinalize() {
		try {
			if (this.test != null) {
				runAfterMethods();
			}
		} catch (Throwable t) {
			reportError(t);
		} finally {
			if (this.test != null) {
				reportTestEnd();
			}
			this.testMethod = null;
		}
	}

	private synchronized void wrapperRunTestMethods() {
		try {
			try {
				boolean ok = wrapperRunBeforeClassMethod();
				if (ok) {
					runTestMethods();
				}
			} finally {
				wrapperRunAfterClassMethod();
			}
		} catch (Throwable t) {
			reportError(t);
		}
	}

	@Override
	public String getTestClass() {
		return this.testClassName;
	}

	@Override
	public synchronized boolean run(TestListener listener) {
		this.errors = this.failures = 0;
		this.testListener = listener;
		if (this.testListener != null) {
			this.testListener.onSuiteStart(this.testClassName);
		}
		wrapperRunTestMethods();
		if (this.testListener != null) {
			this.testListener.onSuiteEnd(this.testClassName, this.runs, this.failures, this.errors);
		}
		return this.errors == 0 && this.failures == 0;
	}

}
