/*
 * Copyright 2023-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.fp.widget.display.brs;

import ej.fp.widget.display.DisplayBufferManager;
import ej.microui.display.Rectangle;

/**
 * The strategy "legacy" (equivalent to the one available in UI Packs 13.x) is dedicated to the buffer mode SWAP (see
 * MicroEJ documentation).
 * <p>
 * It consists in swapping the back buffer and the display front buffer at flush time and to let the BSP restore itself
 * the back buffer with the content of the previous drawings (the past) before unlocking the Graphics Engine after a
 * flush. By consequence, the past is always available before making the very first drawing after a flush.
 */
public class LegacyRefreshStrategy implements BufferRefreshStrategy {

	private Rectangle[] dirtyRectangles;
	private int currentDirtyRectangle;

	@Override
	public void setDisplaySize(int width, int height) {
		// Not used.
	}

	@Override
	public void setBufferCount(int bufferCount) {
		int count = Math.max(bufferCount - 1, 1);
		this.dirtyRectangles = new Rectangle[count];
		for (int i = 0; i < count; i++) {
			this.dirtyRectangles[i] = new Rectangle(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MIN_VALUE,
					Integer.MIN_VALUE);
		}
	}

	@Override
	public void newDrawingRegion(DisplayBufferManager bufferManager, Rectangle rectangle, boolean drawingNow) {
		updateBoundingBox(rectangle);
	}

	private void updateBoundingBox(Rectangle rectangle) {
		int x1 = rectangle.getX1();
		int x2 = rectangle.getX2();
		int y1 = rectangle.getY1();
		int y2 = rectangle.getY2();
		for (int i = 0; i < this.dirtyRectangles.length; i++) {
			Rectangle dirtyRectangle = this.dirtyRectangles[i];
			dirtyRectangle.setX1(Math.min(dirtyRectangle.getX1(), x1));
			dirtyRectangle.setY1(Math.min(dirtyRectangle.getY1(), y1));
			dirtyRectangle.setX2(Math.max(dirtyRectangle.getX2(), x2));
			dirtyRectangle.setY2(Math.max(dirtyRectangle.getY2(), y2));
		}
	}

	@Override
	public void refresh(DisplayBufferManager bufferManager) {
		Rectangle dirtyRectangleBuffer = this.dirtyRectangles[this.currentDirtyRectangle];
		bufferManager.flush(new Rectangle[] { new Rectangle(dirtyRectangleBuffer) });
		// Restore the encapsulating bounds to the other buffer.
		bufferManager.restore(dirtyRectangleBuffer);
		initializeBoundingBox(dirtyRectangleBuffer);
		this.currentDirtyRectangle = (this.currentDirtyRectangle + 1) % (this.dirtyRectangles.length);
	}

	private void initializeBoundingBox(Rectangle rectangle) {
		rectangle.setX1(Integer.MAX_VALUE);
		rectangle.setY1(Integer.MAX_VALUE);
		rectangle.setX2(Integer.MIN_VALUE);
		rectangle.setY2(Integer.MIN_VALUE);
	}

}
