/*
 * Java
 *
 * Copyright 2018-2019 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.junit;

/**
 * Executes all the test methods in all the test classes.
 */
public abstract class AbstractAllTestClassesWrapper implements TestWrapper {

	private int runs;
	private int failures;
	private int errors;
	/** The test listener notified when the tests are executed. */
	protected TestListener testListener;

	/**
	 * Class {@link #wrapperRunTestClass(AbstractTestWrapper)} for each {@link AbstractTestWrapper} to execute.
	 */
	protected abstract void runTestClasses();

	/**
	 * Runs all the methods in the test.
	 *
	 * @param test
	 *            the test.
	 */
	protected synchronized void wrapperRunTestClass(AbstractTestWrapper test) {
		try {
			try {
				test.errors = test.failures = 0;
				test.testListener = this.testListener;
				boolean ok = test.wrapperRunBeforeClassMethod();
				if (ok) {
					test.runTestMethods();
				}
			} finally {
				this.runs += test.runs;
				this.errors += test.errors;
				this.failures += test.failures;
				test.wrapperRunAfterClassMethod();
			}
		} catch (Throwable t) {
			test.reportError(t);
		}
	}

	@Override
	public String getTestClass() {
		return getClass().getName();
	}

	@Override
	public synchronized boolean run(TestListener listener) {
		this.errors = this.failures = 0;
		this.testListener = listener;
		if (this.testListener != null) {
			this.testListener.testSuiteStarted(getTestClass());
		}
		runTestClasses();
		if (this.testListener != null) {
			this.testListener.testSuiteEnded(getTestClass(), this.runs, this.failures, this.errors);
		}
		return this.errors == 0 && this.failures == 0;
	}

}
