/*
 * Java
 *
 * Copyright 2009-2025 MicroEJ Corp. All rights reserved.
 * MicroEJ Corp. PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package com.is2t.hil;

import java.io.Closeable;

import ej.sni.NativeResource;


/**
 * <p>
 * This class provides some utilities to implement simulated natives.
 * <ul>
 * <li>It allows to refresh/flush the content of arrays passed as arguments through HIL protocol. Synchronization is
 * done manually to limit byte transfer (speed optimization). An immutable array can be synchronized (refresh/flush) at
 * any time. An array passed as argument is local to the invoke context (new instance). It cannot be refreshed but it
 * may be flushed when the method that has created this argument returns.</li>
 * <li>It provides interaction to get back resources content</li>
 * <li>It also gives a framework to interact with the shutdown process of the whole simulation thanks to
 * {@link StopListener} and the {@link #stop()} method</li>
 * </ul>
 */
public interface NativeInterface {

	/**
	 * Tells whether an object is an immortal array or not.
	 *
	 * @param obj
	 *            the object to test.
	 * @return <code>true</code> if the given object is an immortal array, <code>false</code> otherwise.
	 */
	boolean isImmortal(Object obj);

	/**
	 * Refresh full content of array if it is immortal.<br>
	 * Does nothing if the given object is not an immortal array.<br>
	 * Equivalent to call {@link #refreshImmortalContent(Object)} assuming that the array is immortal.
	 *
	 * @see #isImmortal(Object)
	 */
	void refreshContent(Object array);

	/**
	 * Refresh content of array if it is immortal.<br>
	 * Does nothing if the given object is not an immortal array. Equivalent to call
	 * {@link #refreshImmortalContent(Object, int, int)} assuming that the array is immortal.
	 *
	 * @see #refreshImmortalContent(Object, int, int)
	 * @see #isImmortal(Object)
	 */
	void refreshContent(Object array, int offset, int length);

	/**
	 * Refresh fully an immortal array.<br>
	 * Equivalent to call {@link #refreshImmortalContent(Object, int, int)} with the full array range.
	 */
	void refreshImmortalContent(Object array);

	/**
	 * Refresh content of an immortal array.<br>
	 * Only range <code>[offset, offset + length - 1]</code> is refreshed.
	 *
	 * @param array
	 *            the immortal array to refresh.
	 * @param offset
	 *            the start index of the range.
	 * @param length
	 *            the length of the range.
	 * @throws IllegalArgumentException
	 *             if the given array is not an array or is not immortal.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given range is invalid.
	 * @throws BrokenConnection
	 *             if the array cannot be refreshed.
	 */
	void refreshImmortalContent(Object array, int offset, int length);

	/**
	 * Flush fully an array.<br>
	 * Equivalent to call {@link #flushContent(Object, int, int)} with the full array range.
	 */
	void flushContent(Object array);

	/**
	 * Flush the content of an array.<br>
	 * Only range <code>[offset, offset + length - 1]</code> is flushed. If array is immortal, this method calls
	 * {@link #flushImmortalContent(Object, int, int)}. Otherwise, if array has been passed as argument of an invoke
	 * method, the content of the array is marked to be flushed when method returns. This method may be called many
	 * times to extend the range to flush (union of intervals).
	 *
	 * @param array
	 *            the immortal array to flush.
	 * @param offset
	 *            the start index of the range.
	 * @param length
	 *            the length of the range.
	 * @throws IllegalArgumentException
	 *             if the given array is not an array or is not immortal.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given range is invalid.
	 * @throws BrokenConnection
	 *             if the array cannot be flushed.
	 */
	void flushContent(Object array, int offset, int length);

	/**
	 * Flush fully an immortal array.<br>
	 * Equivalent to call {@link #flushImmortalContent(Object, int, int)} with the full array range.
	 */
	void flushImmortalContent(Object array);

	/**
	 * Flush content of an immortal array.<br>
	 * Only range <code>[offset, offset + length - 1]</code> is flushed.
	 *
	 * @param array
	 *            the immortal array to flush.
	 * @param offset
	 *            the start index of the range.
	 * @param length
	 *            the length of the range.
	 * @throws IllegalArgumentException
	 *             if the given array is not an array or is not immortal.
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the given range is invalid.
	 * @throws BrokenConnection
	 *             if the array cannot be flushed.
	 */
	void flushImmortalContent(Object array, int offset, int length);

	/**
	 * Get full content of a resource.
	 *
	 * @param resourceName
	 *            as described in {@link Class#getResourceAsStream(String)}.
	 * @return the content of the resource (byte array) or <code>null</code> if not found.
	 * @throws IllegalArgumentException
	 *             if the given resource name is not absolute.
	 * @throws BrokenConnection
	 *             if the resource could not be retrieved.
	 */
	byte[] getResourceContent(String resourceName);

	/**
	 * Register a stop listener.<br>
	 * The stop listeners will be notified when the simulation stops.
	 *
	 * @param stopListener
	 *            the listener to register.
	 */
	void addStopListener(StopListener stopListener);

	/**
	 * Allows to request the simulation stop.
	 */
	void stop();

	/**
	 * Transforms a Java String into a C String.<br>
	 * The platform default encoding is used to transform Java characters into C characters.<br>
	 * The created C String is a NULL terminated String (ends with '\0'). The <code>cString</code> array length
	 * must be at least <code>javaString.length()+1</code>.<br>
	 * The content of the given <code>cString</code> array is automatically flushed with the method {@link NativeInterface#flushContent(Object, int, int)}.
	 * Only <code>javaString.length()</code> bytes are flushed.
	 *
	 * @param javaString the Java String
	 * @param cString byte array which contains the C String.
	 *
	 * @throws IllegalArgumentException if javaString or cString is null
	 * @throws ArrayIndexOutOfBoundsException if cString is too small to contain the string
	 */
	void toCString(String javaString, byte[] cString);

	/**
	 * Transforms a C String into a Java String, using platform default encoding.
	 * The C String must be NULL terminated.<br>
	 * The content of the given <code>cString</code> is automatically refreshed with
	 * the method {@link NativeInterface#refreshContent(Object)}.
	 *
	 * @param cString
	 *            byte array which contains the C String
	 * @return a new Java String.
	 *
	 * @throws IllegalArgumentException
	 *             if cString is null or its length is 1
	 * @throws IllegalArgumentException
	 *             if cString is not NULL terminated.
	 */
	String toJavaString(byte[] cString);

	/**
	 * Registers a native resource that will be closed automatically when the simulator will stop or when
	 * the native resource will be unregistered (see {@link #unregisterNativeResource(int)}).
	 * When the native resource is closed, the <code>close()</code> method is called on the given {@link Closeable}.
	 * <p>
	 * The native resource can also be closed if the application calls the SNI method <code>NativeResource.closeOnGC()</code>
	 * and if the object linked to the native resource is reclaimed by the application garbage collector.
	 *
	 * @param resource the resource to register.
	 * @return the {@link NativeResource} for the resource.
	 */
	NativeResource registerNativeResource(Closeable resource);

	/**
	 * Unregisters the given resource. This method doesn't call the close method on the {@link NativeResource}.
	 * The given resourceId must be a value returned by {@link NativeResource#getResourceId()}.
	 *
	 * @param resourceId ID of the resource.
	 *
	 * @return the {@link NativeResource} that has been unregistered.
	 *
	 * @throws IllegalArgumentException if no native resource is registered with the given ID.
	 */
	NativeResource unregisterNativeResource(int resourceId) throws IllegalArgumentException;

	/**
	 * Returns the registered {@link NativeResource} for the given resource ID.
	 * The given resource ID must be a value returned by {@link NativeResource#getResourceId()}.
	 *
	 * @param resourceId ID of the resource.
	 * @return the native resource.
	 * @throws IllegalArgumentException if no native resource is registered with the given ID.
	 */
	NativeResource getNativeResource(int resourceId) throws IllegalArgumentException;

	/**
	 * Returns an integer that can be passed by the application to the method <code>SNI.closeOnGC()</code> as
	 * the <code>closeFunction</code> argument.
	 */
	int getNativeResourceCloseFunctionId();

	/**
	 * Notifies the simulator that the current native is suspended so it can schedule a thread with
	 * a lower priority.<p>
	 * This method must be called before a wait or a sleep to simulate that a native has been suspended in SNI.
	 */
	void notifySuspendStart();

	/**
	 * Notifies the simulator that the current native is no more suspended. Lower priority threads in the
	 * simulator will not be scheduled anymore.<p>
	 * This method must be called after a wait or a sleep to simulate that a native has been resumed in SNI.
	 */
	void notifySuspendEnd();
}
