/*
 * Java
 *
 * Copyright 2018-2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.bluetooth;

import ej.annotation.Nullable;
import ej.bluetooth.listeners.ConnectionListener;

/**
 * The <code>BluetoothAdapter</code> class represents a Bluetooth adapter.
 * <p>
 * The adapter of the system may be retrieved by calling {@link #getAdapter() getAdapter()}. The GAP events related to
 * the adapter may be handled by {@link #setConnectionListener(ConnectionListener) setting the connection listener} of
 * the adapter. The adapter can be {@link #enable() enabled} or {@link #disable() disabled}.
 * <p>
 * A Bluetooth adapter is able to scan nearby devices and to initiate a connection with a device. It is also able to
 * send advertisements to nearby devices. GATT services may be added to the adapter so that they can be used by
 * connected devices. This class also provides a method to retrieve the list of active connections.
 */
public class BluetoothAdapter {

	/**
	 * Private constructor.
	 */
	private BluetoothAdapter() {
		throw new RuntimeException();
	}

	/**
	 * Returns the adapter of the system. The adapter has to be {@link #enable() enabled} before being used.
	 *
	 * @return the adapter of the system.
	 */
	public static BluetoothAdapter getAdapter() {
		throw new RuntimeException();
	}

	/**
	 * Enables this adapter.
	 * <p>
	 * If this adapter is already enabled, this method returns {@code true} immediately.
	 *
	 * @return true if this adapter was enabled successfully, false otherwise.
	 */
	public boolean enable() {
		throw new RuntimeException();
	}

	/**
	 * Disables this adapter. This method frees up all resources allocated by this adapter, including all the services
	 * added using {@link #addService addService()}.
	 * <p>
	 * If this adapter is already disabled, calling this method has no effect.
	 */
	public void disable() {
		throw new RuntimeException();
	}

	/**
	 * Adds the given service to this adapter.
	 *
	 * @param serviceDefinition
	 *            the definition of the service to add.
	 * @return true if the service was added successfully, false otherwise.
	 */
	public @Nullable BluetoothService addService(BluetoothServiceDefinition serviceDefinition) {
		throw new RuntimeException();
	}

	/**
	 * Sets the listener which will receive the GAP events related to this adapter.
	 * <p>
	 * Only one listener may be set at the same time. If the connection listener has not been set, the
	 * {@link ej.bluetooth.listeners.impl.DefaultConnectionListener} will be used to handle the events.
	 *
	 * @param connectionListener
	 *            the object which will receive the events.
	 */
	public void setConnectionListener(ConnectionListener connectionListener) {
		throw new RuntimeException();
	}

	/**
	 * Starts advertising. If a device connects while advertising, the advertisement is stopped. It may be started again
	 * once the connection is established.
	 *
	 * @param advertisementData
	 *            the advertisement data.
	 * @return true if advertising was started successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onAdvertisementCompleted()
	 * @see ej.bluetooth.listeners.ConnectionListener#onConnected(BluetoothConnection)
	 */
	public boolean startAdvertising(byte[] advertisementData) {
		throw new RuntimeException();
	}

	/**
	 * Stops advertising.
	 *
	 * @return true if advertising was stopped successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onAdvertisementCompleted()
	 */
	public boolean stopAdvertising() {
		throw new RuntimeException();
	}

	/**
	 * Starts scanning. If a connection is initiated while scanning, the scan is stopped. It may be started again once
	 * the connection is established. The given scan filter is applied to keep or discard scan results.
	 *
	 * @param scanFilter
	 *            the scan filter.
	 * @return true if scanning was started successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onScanResult(BluetoothAddress, byte[], int)
	 * @see ej.bluetooth.listeners.ConnectionListener#onScanCompleted()
	 */
	public boolean startScanning(BluetoothScanFilter scanFilter) {
		throw new RuntimeException();
	}

	/**
	 * Stops scanning.
	 *
	 * @return true if scanning was stopped successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onScanCompleted()
	 */
	public boolean stopScanning() {
		throw new RuntimeException();
	}

	/**
	 * Initiates a connection to the device with the given address.
	 *
	 * @param address
	 *            the address of the device to connect to.
	 *
	 * @return true if connection was initiated successfully, false otherwise.
	 * @see ej.bluetooth.listeners.ConnectionListener#onConnectFailed(BluetoothAddress)
	 * @see ej.bluetooth.listeners.ConnectionListener#onConnected(BluetoothConnection)
	 */
	public boolean connect(BluetoothAddress address) {
		throw new RuntimeException();
	}
}
