/*
 * Copyright 2015-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.style.outline.border;

import ej.annotation.Nullable;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.mwt.style.outline.FlexibleOutline;
import ej.mwt.util.Size;

/**
 * Represents and draw a rectangular border with different thicknesses and colors.
 * <p>
 * Each edge color can be customized.
 * <p>
 * The thicknesses are stored as a <code>char</code> for heap optimization and therefore cannot exceed
 * <code>65535</code>.
 */
public class FlexibleRectangularMulticoloredBorder extends FlexibleOutline {

	private final int colorTop;
	private final int colorBottom;
	private final int colorLeft;
	private final int colorRight;

	/**
	 * Creates a flexible rectangular border specifying its edges thickness and color.
	 * <p>
	 * The given thickness values are clamped between <code>0</code> and <code>Character.MAX_VALUE</code>.
	 *
	 * @param top
	 *            the top thickness.
	 * @param colorTop
	 *            the top color.
	 * @param right
	 *            the right thickness.
	 * @param colorRight
	 *            the right color.
	 * @param bottom
	 *            the bottom thickness.
	 * @param colorBottom
	 *            the bottom color.
	 * @param left
	 *            the left thickness.
	 * @param colorLeft
	 *            the left color.
	 */
	public FlexibleRectangularMulticoloredBorder(int top, int colorTop, int right, int colorRight, int bottom,
			int colorBottom, int left, int colorLeft) {
		super(top, right, bottom, left);
		this.colorTop = colorTop;
		this.colorRight = colorRight;
		this.colorBottom = colorBottom;
		this.colorLeft = colorLeft;
	}

	/**
	 * Gets the top color.
	 *
	 * @return the top color.
	 */
	public int getColorTop() {
		return this.colorTop;
	}

	/**
	 * Gets the bottom color.
	 *
	 * @return the bottom color.
	 */
	public int getColorBottom() {
		return this.colorBottom;
	}

	/**
	 * Gets the left color.
	 *
	 * @return the left color.
	 */
	public int getColorLeft() {
		return this.colorLeft;
	}

	/**
	 * Gets the right color.
	 *
	 * @return the right color.
	 */
	public int getColorRight() {
		return this.colorRight;
	}

	@Override
	public void apply(GraphicsContext g, Size size) {
		int width = size.getWidth();
		int height = size.getHeight();

		g.setColor(this.colorTop);
		int topThickness = getTop();
		Painter.fillRectangle(g, 0, 0, width, topThickness);

		g.setColor(this.colorBottom);
		int bottomThickness = getBottom();
		Painter.fillRectangle(g, 0, height - bottomThickness, width, bottomThickness);

		g.setColor(this.colorLeft);
		int leftThickness = getLeft();
		Painter.fillRectangle(g, 0, 0, leftThickness, height);

		g.setColor(this.colorRight);
		int rightThickness = getRight();
		Painter.fillRectangle(g, width - rightThickness, 0, rightThickness, height);

		super.apply(g, size);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			FlexibleRectangularMulticoloredBorder border = (FlexibleRectangularMulticoloredBorder) obj;
			return super.equals(obj) && this.colorTop == border.colorTop && this.colorBottom == border.colorBottom
					&& this.colorLeft == border.colorLeft && this.colorRight == border.colorRight;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return super.hashCode() * this.colorTop * this.colorBottom * this.colorLeft * this.colorRight;
	}

}
