/*
 * Copyright 2010-2015 http://beders.github.com/Resty
 * Copyright 2015-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *        http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ej.rest.web;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URI;

import org.json.me.JSONArray;
import org.json.me.JSONException;
import org.json.me.JSONObject;
import org.json.me.JSONTokener;


/** A resource presentation in JSON format.
 * You can  ask Resty to parse the JSON into a JSONArray or a JSONObject. The JSONObject is similar to org.json.JSONObject
 * and allows full access to the JSON.
 * You can also access the JSON with a JSONPathQuery to extract only the parts you are interested in.
 * 
 * @author beders
 * @author RobertFischer
 */
public class JSONResource extends AbstractResource {

	Object json;

	public JSONResource(Option... options) {
		super(options);
	}

	/**
	 * Parse and return JSON array. Parsing is done only once after which the inputStream is at EOF.
	 */
	public JSONArray array() throws IOException, JSONException {
		if(json == null) {
			unmarshal();
		}
		return (JSONArray)json;
	}

	/**
	 * Parse and return JSON object. Parsing is done only once after which the inputStrem is at EOF.
	 * @return the JSON object
	 * @throws IOException
	 * @throws JSONException
	 */
	public JSONObject object() throws IOException, JSONException {
		if (json == null) {
			unmarshal();
		}
		return (JSONObject)json;
	}

	/** Added for compatibility with Scala. See Issue #2 at github.
	 * 
	 * @return the JSONObject presentation
	 * @throws IOException
	 * @throws JSONException if data was no valid JSON
	 */
	public JSONObject toObject() throws IOException, JSONException {
		return object();
	}

	/** Transforming the JSON */
	protected Object unmarshal() throws IOException, JSONException {

		final char[] buffer = new char[16];
		final StringBuilder out = new StringBuilder();
		Reader in = new InputStreamReader(inputStream, "UTF-8");
		for (;;) {
			int rsz = in.read(buffer, 0, buffer.length);
			if (rsz < 0) {
				break;
			}
			out.append(buffer, 0, rsz);
		}

		json = new JSONTokener(out.toString()).nextValue();
		inputStream.close();
		return json;
	}

	/** Execute the given path query on the json GET the returned URI expecting JSON
	 * 
	 * @param path path to the URI to follow
	 * @return a new resource, as a result of getting it from the server in JSON format
	 * @throws Exception
	 * @throws JSONException
	 */
	public JSONResource json(JSONPathQuery path) throws Exception {
		Object jsonValue = path.eval(this);
		return json(jsonValue.toString());
	}

	/** Execute the given path query on the json and POST to the returned URI expecting JSON
	 * 
	 * @param path path to the URI to follow
	 * @return a new resource, as a result of getting it from the server in JSON format
	 * @throws Exception
	 * @throws JSONException
	 */
	public JSONResource json(JSONPathQuery path, Content content) throws Exception {
		Object jsonValue = path.eval(this);
		return json(jsonValue.toString(), content);
	}

	/** Execute the given path query on the json and use the returned string as an URI expecting text/*
	 * 
	 * @param path path to the URI to follow
	 * @return a new resource, as a result of getting it from the server in text/plain format
	 * @throws Exception
	 * @throws JSONException
	 */
	public TextResource text(JSONPathQuery path) throws Exception {
		Object jsonValue = path.eval(this);
		return text(URI.create(jsonValue.toString()));
	}

	/** Gets the partial JSON object or attribute as specified in the path expression.*/
	public Object get(String path) throws Exception {
		return new JSONPathQuery(path).eval(this);
	}

	/** Gets the partial JSON object or attribute as specified in the path expression.*/
	public Object get(JSONPathQuery aQuery) throws Exception {
		return aQuery.eval(this);
	}

	@Override
	public String getAcceptedTypes() {
		return "application/json";
	}

}
