/*
 * Java
 *
 * Copyright 2009-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.hoka.http.encoding;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import ej.hoka.http.HttpServer;

/**
 * Interface for defining HTTP encoding handlers. A HTTP encoding handler is able to decode data from an
 * {@link InputStream} and encode data to an {@link OutputStream}.
 * <p>
 * Encodings are specified in HTTP headers such as <code>content-encoding</code>, <code>transfer-encoding</code>,
 * <code>accept-encoding</code>.
 * <p>
 * Encoding handlers should be registered in the {@link HttpServer} in order to use them.
 *
 * @see EncodingRegistry#registerContentEncoding(ContentEncoding)
 *
 */
public interface ContentEncoding {

	/**
	 * Returns the name of the supported encoding.
	 *
	 * @return an internal {@link String} in lower case format.
	 */
	String getId();

	/**
	 * Returns an {@link InputStream} to read the decoded data from the <code>original</code> {@link InputStream}.
	 *
	 * @param original
	 *            the {@link InputStream} to read the encoded data.
	 * @return the {@link InputStream} to read the decoded data.
	 * @throws IOException
	 *             if any I/O error occurs.
	 */
	InputStream open(InputStream original) throws IOException;

	/**
	 * Wraps the <code>original</code> {@link OutputStream} with a special {@link OutputStream} which performs the
	 * encoding. Returns an {@link OutputStream} to encode the data from the <code>original</code> {@link OutputStream}.
	 *
	 * @param original
	 *            the output stream to wrap.
	 * @return the {@link OutputStream} to encode the data.
	 * @throws IOException
	 *             if any I/O error occurs.
	 */
	OutputStream open(OutputStream original) throws IOException;

}
