/*
 * JACOB - CBOR implementation in Java.
 *
 * (C) Copyright - 2013 - J.W. Janssen <j.w.janssen@lxtreme.nl>
 * Copyright 2016-2019 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed under Apache License v2.0.
 */
package ej.cbor;

import static ej.cbor.CborConstants.TYPE_ARRAY;
import static ej.cbor.CborConstants.TYPE_BYTE_STRING;
import static ej.cbor.CborConstants.TYPE_FLOAT_SIMPLE;
import static ej.cbor.CborConstants.TYPE_MAP;
import static ej.cbor.CborConstants.TYPE_NEGATIVE_INTEGER;
import static ej.cbor.CborConstants.TYPE_TAG;
import static ej.cbor.CborConstants.TYPE_TEXT_STRING;
import static ej.cbor.CborConstants.TYPE_UNSIGNED_INTEGER;


/**
 * Represents the various major types in CBOR, along with their .
 * <p>
 * The major type is encoded in the upper three bits of each initial byte. The lower 5 bytes represent any additional information.
 * </p>
 */
public class CborType {
    private static final ErrorMessage ERR = ErrorMessage.getDefault();

    private final int major;
    private final int additional;

    private CborType(int major, int additional) {
        this.major = major;
        this.additional = additional;
    }

    /**
     * Returns a descriptive string for the given major type.
     *
     * @param mt the major type to return as string, values from [0..7] are supported.
     * @return the name of the given major type, as String, never <code>null</code>.
     * @throws IllegalArgumentException in case the given major type is not supported.
     */
    public static String getName(int mt) throws IllegalArgumentException {
        switch (mt) {
            case TYPE_ARRAY:
                return "array";
            case TYPE_BYTE_STRING:
                return "byte string";
            case TYPE_FLOAT_SIMPLE:
                return "float/simple value";
            case TYPE_MAP:
                return "map";
            case TYPE_NEGATIVE_INTEGER:
                return "negative integer";
            case TYPE_TAG:
                return "tag";
            case TYPE_TEXT_STRING:
                return "text string";
            case TYPE_UNSIGNED_INTEGER:
                return "unsigned integer";
            default:
                throw new IllegalArgumentException(ERR.messageAt(ErrorMessage.INVALID_MAJOR_TYPE, mt));
        }
    }

    /**
     * Decodes a given byte value to a {@link CborType} value.
     *
     * @param i the input byte (8-bit) to decode into a {@link CborType} instance.
     * @return a {@link CborType} instance, never <code>null</code>.
     */
    public static CborType valueOf(int i) {
        return new CborType((i & 0xff) >>> 5, i & 0x1f);
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null || getClass() != obj.getClass()) {
            return false;
        }

        CborType other = (CborType) obj;
        return (this.major == other.major) && (this.additional == other.additional);
    }

    /**
     * @return the additional information of this type, as integer value from [0..31].
     */
    public int getAdditionalInfo() {
        return this.additional;
    }

    /**
     * @return the major type, as integer value from [0..7].
     */
    public int getMajorType() {
        return this.major;
    }

    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + this.additional;
        result = prime * result + this.major;
        return result;
    }

    /**
     * @return <code>true</code> if this type allows for an infinite-length payload,
     *         <code>false</code> if only definite-length payloads are allowed.
     */
    public boolean isBreakAllowed() {
        return this.major == TYPE_ARRAY || this.major == TYPE_BYTE_STRING || this.major == TYPE_MAP
                        || this.major == TYPE_TEXT_STRING;
    }

    /**
     * Determines whether the major type of a given {@link CborType} equals the major type of this {@link CborType}.
     *
     * @param other the {@link CborType} to compare against, cannot be <code>null</code>.
     * @return <code>true</code> if the given {@link CborType} is of the same major type as this {@link CborType}, <code>false</code> otherwise.
     * @throws IllegalArgumentException in case the given argument was <code>null</code>.
     */
    public boolean isEqualType(CborType other) throws IllegalArgumentException {
        if (other == null) {
            throw new IllegalArgumentException(ERR.messageAt(ErrorMessage.PARAMETERS_CANNOT_BE_NULL));
        }
        return this.major == other.major;
    }

    /**
     * Determines whether the major type of a given byte value (representing an encoded {@link CborType}) equals the major type of this {@link CborType}.
     *
     * @param encoded the encoded CBOR type to compare.
     * @return <code>true</code> if the given byte value represents the same major type as this {@link CborType}, <code>false</code> otherwise.
     */
    public boolean isEqualType(int encoded) {
        return this.major == ((encoded & 0xff) >>> 5);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append(getName(this.major)).append('(').append(this.additional).append(')');
        return sb.toString();
    }
}
