/*
 * Java
 *
 * Copyright 2014-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.websocket;

/**
 * An {@link Endpoint} knows how to handle high level websocket events. It is responsible for the applicative reactions.
 * <p>
 * An {@link Endpoint} defines callbacks that a {@link WebSocket} can use to handle high level events.
 *
 *
 *
 */
public interface Endpoint {

	/**
	 * A binary message has been received.
	 * <p>
	 * If the return value is not null, then it is sent to the remote endpoint.
	 *
	 * @param ws
	 *            the {@link WebSocket} on which the event occurred
	 * @param message
	 *            the received message
	 * @return a return message that the WebSocket will send to the remote endpoint
	 */
	byte[] onBinaryMessage(WebSocket ws, byte[] message);

	/**
	 * The associated {@link WebSocket} is about to close. This method is called if the remote endpoint started a closing handshake (that is, a
	 * close control frame has been received and connection state was OPEN), or if the local application request to close the connection.
	 * <p>
	 * The underlying socket connection may already be closed to you should not try to send any further message in this callback.
	 *
	 * @param ws
	 *            the {@link WebSocket} about to be closed
	 * @param closeReason
	 *            the reason for closing
	 */
	void onClose(WebSocket ws, ReasonForClosure closeReason);

	/**
	 * An error occurred on the {@link WebSocket}.
	 *
	 * @param ws
	 *            the {@link WebSocket} on which the event occurred
	 * @param thr
	 *            the original error
	 */
	void onError(WebSocket ws, Throwable thr);


	/**
	 * The connection has been established.
	 *
	 * @param ws
	 *            the {@link WebSocket} on which the event occurred
	 */
	void onOpen(WebSocket ws);

	/**
	 * A pong frame has been received.
	 *
	 * @param data
	 *            the data received in the pong payload
	 */
	void onPong(byte[] data);

	/**
	 * A ping frame has been received. The pong frame does not need to be sent
	 * manually, it will be done automatically by the websocket.
	 *
	 * @param data the data received in the ping payload
	 */
	void onPing(byte[] data);

	/**
	 * A text message has been received.
	 * <p>
	 * If the return value is not null, then it is sent to the remote endpoint.
	 *
	 * @param ws
	 *            the {@link WebSocket} on which the event occurred
	 * @param message
	 *            the received message
	 * @return a return message that the WebSocket will send to the remote endpoint
	 */
	String onTextMessage(WebSocket ws, String message);

}
