/*
 * Copyright 2024-2025 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package com.microej.wear.components;

import ej.microui.display.GraphicsContext;

/**
 * Represents a UI that can be rendered on a graphics context and that can be shown on the display.
 * <p>
 * The methods of a renderable are called in the following order:
 * <ol>
 * <li>one call to {@link #onAttached}</li>
 * <li>any number of calls to {@link #render}</li>
 * <li>one optional call to {@link #showOnDisplay}</li>
 * <li>one call to {@link #onDetached}</li>
 * </ol>
 */
public interface Renderable {

	/**
	 * Called when this instance is scheduled for imminent rendering.
	 * <p>
	 * For example, this method can load the resources necessary for rendering.
	 */
	void onAttached();

	/**
	 * Called when this instance is no longer scheduled for imminent rendering.
	 * <p>
	 * For example, this method can dispose the resources loaded in {@link #onAttached()}.
	 */
	void onDetached();

	/**
	 * Renders this instance on the given graphics context.
	 * <p>
	 * This method applies the translation and clip set in the given graphics context (it should not reset the
	 * translation and clip before drawing).
	 *
	 * @param g
	 *            the graphics context to draw with.
	 */
	void render(GraphicsContext g);

	/**
	 * Shows this instance on the display.
	 * <p>
	 * Once the displayable is shown on the display, it can start animations and handle events. Once it is hidden (due
	 * to an other displayable being shown), it can stop the started animations and dispose the resources loaded in
	 * {@link #onAttached()}.
	 */
	void showOnDisplay();
}
