/*
 * Java
 *
 * Copyright 2018-2020 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.bluetooth.listeners;

import ej.bluetooth.BluetoothAttribute;
import ej.bluetooth.BluetoothCharacteristic;
import ej.bluetooth.BluetoothConnection;

/**
 * The <code>LocalServiceListener</code> interface provides callbacks related to local services.
 * <p>
 * After {@link ej.bluetooth.BluetoothAdapter#addService adding a service} to the adapter, the application may
 * {@link ej.bluetooth.BluetoothService#setLocalListener set the local listener} of a service in order to be notified of
 * the events related to this service.
 * <p>
 * This library provides a default implementation of this interface:
 * {@link ej.bluetooth.listeners.impl.DefaultLocalServiceListener}.
 */
public interface LocalServiceListener {

	/**
	 * Called when a read request is received.
	 *
	 * @param connection
	 *            the connection with the device which has sent the request.
	 * @param attribute
	 *            the attribute to read.
	 */
	void onReadRequest(BluetoothConnection connection, BluetoothAttribute attribute);

	/**
	 * Called when a write request is received.
	 *
	 * @param connection
	 *            the connection with the device which has sent the request.
	 * @param attribute
	 *            the attribute to write.
	 * @param value
	 *            the value to write.
	 */
	void onWriteRequest(BluetoothConnection connection, BluetoothAttribute attribute, byte[] value);

	/**
	 * Called when a characteristic notification or indication is sent.
	 * <p>
	 * If a notification was sent, this method is called as soon as the notification is sent to the device. If an
	 * indication was sent, this method is called when the device acknowledges the indication.
	 *
	 * @param connection
	 *            the connection with the device to which the notification was sent.
	 * @param characteristic
	 *            the characteristic which has changed.
	 * @param success
	 *            true if the notification was sent successfully, false otherwise.
	 * @see ej.bluetooth.BluetoothConnection#sendNotification(BluetoothCharacteristic, byte[], boolean)
	 */
	void onNotificationSent(BluetoothConnection connection, BluetoothCharacteristic characteristic, boolean success);
}
