/*
 * Java
 *
 * Copyright 2018-2021 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.aws.iot;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import org.eclipse.paho.client.mqttv3.MqttClient;
import org.eclipse.paho.client.mqttv3.MqttConnectOptions;
import org.eclipse.paho.client.mqttv3.MqttException;

import ej.util.concurrent.SingleThreadExecutor;

/**
 * AWS IoT client, based on MicroPaho MQTT Implementation.
 */
public class AwsIotClient {

	// Shadow constants
	private static final String SLASH = "/"; //$NON-NLS-1$
	private static final String NAME = "/name/"; //$NON-NLS-1$
	private static final byte[] EMPTY_MESSAGE = "{}".getBytes(); //$NON-NLS-1$
	private static final String SHADOW = "/shadow"; //$NON-NLS-1$
	private static final String SHADOW_BASE = "$aws/things/"; //$NON-NLS-1$

	/**
	 * client options.
	 */
	private final AwsIotClientOptions options;

	private MqttClient mqttClient;
	private ExecutorService executorService;
	private final AwsIotConnectionListener connectionListener;
	private Map<String, AwsIotMessageCallback> subscribers;

	/**
	 * Constructor client.
	 *
	 * @param options
	 *            the connection option. use {@link AwsIotClientOptions.Builder} to create an instance of
	 *            {@link AwsIotClientOptions}
	 */
	public AwsIotClient(final AwsIotClientOptions options) {
		this(options, null);
	}

	/**
	 * Constructor client.
	 *
	 * @param options
	 *            the connection option
	 * @param connectionCallback
	 *            connection callback to monitor connection lost events
	 *
	 */
	public AwsIotClient(final AwsIotClientOptions options, final AwsIotConnectionListener connectionCallback) {
		this.options = options;
		this.connectionListener = connectionCallback;
	}

	/**
	 * Connect to AWS IoT service using the client options.
	 *
	 * @throws AwsIotException
	 *             if an error occurred during connection.
	 */
	public void connect() throws AwsIotException {

		// Initialize executor, this is used to dispatch received messages
		ExecutorService executor = this.executorService;
		if (executor == null) {
			executor = new SingleThreadExecutor();
			this.executorService = executor;
		}

		// Initialize subscriber map.
		Map<String, AwsIotMessageCallback> subscribers = this.subscribers;
		if (subscribers == null) {
			subscribers = new HashMap<>(0);
			this.subscribers = subscribers;
		}

		MqttClient mqttClient = this.mqttClient;
		if (mqttClient == null) {
			// 1) this client instance is closed when close is called
			mqttClient = new MqttClient(this.options.getServerURL(), this.options.getClientId()); // NOSONAR see 1)
			mqttClient.setCallback(new AwsIotClientCallback(this));
		}

		final AwsIotClientOptions options = this.options;
		assert options != null;

		final MqttConnectOptions mqttOptions = new MqttConnectOptions();
		mqttOptions.setConnectionTimeout(options.getTimeout());
		mqttOptions.setKeepAliveInterval(options.getKeepAlive());
		mqttOptions.setSocketFactory(options.getSocketFactory());

		// Append SDK name and version for AWS metrics whether or not a username has been set.
		String usernameOrEmpty = options.getUsername() == null ? "" : options.getUsername(); //$NON-NLS-1$
		mqttOptions.setUserName(usernameOrEmpty + "?SDK=MicroEJ&Version=" + SDK.VERSION); //$NON-NLS-1$
		mqttOptions.setPassword(options.getPassword());

		try {

			mqttClient.connect(mqttOptions);
			this.mqttClient = mqttClient;

		} catch (MqttException e) {
			throw new AwsIotException(e);
		}

	}

	/**
	 * Returns the state of the connection.
	 *
	 * @return true if connected, false otherwise
	 */
	public boolean isConnected() {
		final MqttClient mqttClient = this.mqttClient;
		return mqttClient.isConnected();
	}

	/**
	 * Disconnect this client from the underlying communication layer.
	 *
	 * @throws AwsIotException
	 *             when not connected
	 */
	public void disconnect() throws AwsIotException {
		try {
			final MqttClient mqttClient = this.mqttClient;
			assert mqttClient != null;
			mqttClient.disconnect();
		} catch (MqttException e) {
			throw new AwsIotException(e);
		}
	}

	/**
	 * Close this client. This close the underlying MQTT client, clean the subscribers list and shutdown the service
	 * executor which is responsible of executing message callback on a separated thread.
	 *
	 * @throws AwsIotException
	 *             if the client is not disconnected.
	 */
	public void close() throws AwsIotException {
		try {
			final MqttClient mqttClient = this.mqttClient;
			assert mqttClient != null;

			mqttClient.close();

			final ExecutorService executorService = this.executorService;
			assert executorService != null;
			executorService.shutdown();
			this.executorService = null;

			final Map<String, AwsIotMessageCallback> subscribers = this.subscribers;
			assert subscribers != null;
			subscribers.clear();
			this.subscribers = null;

		} catch (MqttException e) {
			throw new AwsIotException(e);
		}
	}

	/**
	 * Adds a topic listener on a topic. Only one listener per topic can exist, if a listener is added on a topic that
	 * already has a listener, it will be replaced. QOS 0 is used
	 *
	 * @param topic
	 *            the topic to add the listener on, one listener is allowed per topic
	 * @param subscriber
	 *            the subscriber that will process the received data
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void subscribe(String topic, AwsIotMessageCallback subscriber) throws AwsIotException {
		subscribe(topic, subscriber, 0);
	}

	/**
	 * Adds a topic listener on a topic. Only one listener per topic can exist, if a listener is added on a topic that
	 * already has a listener, it will be replaced.
	 *
	 * @param topic
	 *            the topic to add the listener on, one listener is allowed per topic
	 * @param subscriber
	 *            the subscriber that will process the received data
	 * @param qos
	 *            MQTT quality of service
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void subscribe(String topic, AwsIotMessageCallback subscriber, int qos) throws AwsIotException {
		try {

			final MqttClient mqttClient = this.mqttClient;
			assert mqttClient != null;
			mqttClient.subscribe(topic, qos);

			final Map<String, AwsIotMessageCallback> subscribers = this.subscribers;
			assert subscriber != null;
			subscribers.put(topic, subscriber);

		} catch (MqttException t) {
			throw new AwsIotException(t);
		}
	}

	/**
	 * Unsubscribe from a topic.
	 *
	 * @param topic
	 *            the topic you want to stop listening to
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void unsubscribe(String topic) throws AwsIotException {
		try {

			final MqttClient mqttClient = this.mqttClient;
			assert mqttClient != null;
			mqttClient.unsubscribe(topic);

			// remove registered message listener for this topic
			final Map<String, AwsIotMessageCallback> subscribers = this.subscribers;
			assert subscribers != null;
			subscribers.remove(topic);

		} catch (MqttException t) {
			throw new AwsIotException(t);
		}
	}

	/**
	 * Publish a message to AWS IoT MQTT broker.
	 *
	 * @param message
	 *            AwsIotMessage
	 * @throws AwsIotException
	 *             on error
	 */
	public void publish(AwsIotMessage message) throws AwsIotException {
		publish(message.getTopic(), message.getPayload(), message.getQos(), message.isRetained());
	}

	/**
	 * Publishes a message on a topic with default QOS 0
	 *
	 * @param topic
	 *            the topic you want to publish on
	 * @param data
	 *            the data you want to publish
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void publish(String topic, byte[] data) throws AwsIotException {
		publish(topic, data, 0);
	}

	/**
	 * Publishes a message on a topic.
	 *
	 * the retained flag is set to false when using this method.
	 *
	 * @param topic
	 *            the topic you want to publish on
	 * @param data
	 *            the data you want to publish
	 *
	 * @param qos
	 *            MQTT quality of service
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void publish(String topic, byte[] data, int qos) throws AwsIotException {
		publish(topic, data, qos, false);
	}

	/**
	 * Publishes a message on a topic.
	 *
	 * @param topic
	 *            the topic you want to publish on
	 * @param data
	 *            the data you want to publish
	 *
	 * @param qos
	 *            MQTT quality of service
	 * @param retained
	 *            set retained flag on the message.
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void publish(String topic, byte[] data, int qos, boolean retained) throws AwsIotException {
		try {

			final MqttClient mqttClient = this.mqttClient;
			assert mqttClient != null;
			mqttClient.publish(topic, data, qos, retained);

		} catch (MqttException t) {
			throw new AwsIotException(t);
		}
	}

	/**
	 * Gets shadow with default QOS 0.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void getShadow() throws AwsIotException {
		getShadow(null);
	}

	/**
	 * Gets shadow.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param qos
	 *            MQTT quality of service
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void getShadow(int qos) throws AwsIotException {
		getShadow(null, qos);
	}

	/**
	 * Gets the named shadow with default QOS 0.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param shadowName
	 *            the shadow name
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void getShadow(String shadowName) throws AwsIotException {
		getShadow(shadowName, 0);
	}

	/**
	 * Gets the named shadow.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param shadowName
	 *            the shadow name
	 *
	 * @param qos
	 *            MQTT quality of service
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void getShadow(String shadowName, int qos) throws AwsIotException {
		String topic = buildShadowTopicName(shadowName, ShadowAction.get, null);
		publish(topic, EMPTY_MESSAGE, qos);
	}

	/**
	 * Creates a no-named shadow if it doesn't exist, or updates the contents of an existing shadow with the state
	 * information provided in the message body.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param message
	 *            the message (body of request)
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void updateShadow(byte[] message) throws AwsIotException {
		updateShadow(null, message);
	}

	/**
	 * Creates a no-named shadow if it doesn't exist, or updates the contents of an existing shadow with the state
	 * information provided in the message body.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param message
	 *            the message (body of request)
	 * @param qos
	 *            MQTT quality of service
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void updateShadow(byte[] message, int qos) throws AwsIotException {
		updateShadow(null, message, qos);
	}

	/**
	 * Creates a shadow if it doesn't exist, or updates the contents of an existing shadow with the state information
	 * provided in the message body.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param shadowName
	 *            the shadow name
	 * @param message
	 *            the message
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void updateShadow(String shadowName, byte[] message) throws AwsIotException {
		updateShadow(shadowName, message, 0);
	}

	/**
	 * updateShadow : Creates a shadow if it doesn't exist, or updates the contents of an existing shadow with the state
	 * information provided in the message body.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param shadowName
	 *            the shadow name
	 * @param message
	 *            the message
	 * @param qos
	 *            MQTT quality of service
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void updateShadow(String shadowName, byte[] message, int qos) throws AwsIotException {
		String topic = buildShadowTopicName(shadowName, ShadowAction.update, null);
		publish(topic, message, qos);
	}

	/**
	 *
	 * Delete the classic shadow of thing with QOS 0.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void deleteShadow() throws AwsIotException {
		deleteShadow(null);
	}

	/**
	 *
	 * Delete the classic shadow of thing.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param qos
	 *            MQTT quality of service
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void deleteShadow(int qos) throws AwsIotException {
		deleteShadow(null, qos);
	}

	/**
	 * Delete the named shadow of thing with QOS 0.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 *
	 * @param shadowName
	 *            the shadow client
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void deleteShadow(String shadowName) throws AwsIotException {
		deleteShadow(shadowName, 0);
	}

	/**
	 * Delete the named shadow of thing.
	 *
	 * Important
	 *
	 * Before calling this method, subscribe to the it's relevant shadow response to receive the response.
	 *
	 * If you do not subscribe to the shadow response before you perform this request, you will not receive the results.
	 *
	 * @see AwsIotClient#subscribeToShadow(ShadowAction, ShadowResult, AwsIotMessageCallback)
	 *
	 * @param shadowName
	 *            the shadow client
	 *
	 * @param qos
	 *            MQTT quality of service
	 *
	 * @throws AwsIotException
	 *             when not connected or on communication error
	 */
	public void deleteShadow(String shadowName, int qos) throws AwsIotException {
		final String topic = buildShadowTopicName(shadowName, ShadowAction.delete, null);
		publish(topic, EMPTY_MESSAGE, qos);
	}

	/**
	 * Subscribe to a shadow action (GET, UPDATE, DELETE) result ACCEPTED, REJECTED and (DELTA, DOCUMENT for update
	 * only).
	 *
	 * This method use QOS 0. to specify the qos use
	 * {@link AwsIotClient#subscribeToShadow(String, ShadowAction, ShadowResult, AwsIotMessageCallback, int)}
	 *
	 * @param action
	 *            the shadow action see {@link ShadowAction} for possible values
	 * @param result
	 *            the result to listen too ACCEPTED, REJECTED and (DELTA, DOCUMENT for update only). Use
	 *            {@link ShadowResult} for possible values
	 * @param callback
	 *            the action callback will be executed when a message is received.
	 *
	 * @throws AwsIotException
	 *             on error.
	 */
	public void subscribeToShadow(ShadowAction action, ShadowResult result, AwsIotMessageCallback callback)
			throws AwsIotException {
		subscribeToShadow(null, action, result, callback);
	}

	/**
	 * Subscribe to a shadow action (GET, UPDATE, DELETE) result ACCEPTED, REJECTED and (DELTA, DOCUMENT for update
	 * only).
	 *
	 * This method use QOS 0. to specify the qos use
	 * {@link AwsIotClient#subscribeToShadow(String, ShadowAction, ShadowResult, AwsIotMessageCallback, int)}
	 *
	 * @param shadowName
	 *            the name of shadow
	 * @param action
	 *            the shadow action see {@link ShadowAction} for possible values
	 * @param result
	 *            the result to listen too ACCEPTED, REJECTED and (DELTA, DOCUMENT for update only). Use
	 *            {@link ShadowResult} for possible values
	 * @param callback
	 *            the action callback will be executed when a message is received.
	 *
	 * @throws AwsIotException
	 *             on error.
	 */
	public void subscribeToShadow(String shadowName, ShadowAction action, ShadowResult result,
			AwsIotMessageCallback callback) throws AwsIotException {

		String topic = buildShadowTopicName(shadowName, action, result);
		subscribe(topic, callback);
	}

	/**
	 * Subscribe to a shadow action (GET, UPDATE, DELETE) result ACCEPTED, REJECTED and (DELTA, DOCUMENT for update
	 * only).
	 *
	 * @param action
	 *            the shadow action see {@link ShadowAction} for possible values
	 * @param result
	 *            the result to listen too ACCEPTED, REJECTED and (DELTA, DOCUMENT for update only). Use
	 *            {@link ShadowResult} for possible values
	 * @param callback
	 *            the action callback will be executed when a message is received.
	 * @param qos
	 *            the MQTT quality of service.
	 * @throws AwsIotException
	 *             on error.
	 */
	public void subscribeToShadow(ShadowAction action, ShadowResult result, AwsIotMessageCallback callback, int qos)
			throws AwsIotException {
		subscribeToShadow(null, action, result, callback, qos);
	}

	/**
	 * Subscribe to a shadow action (GET, UPDATE, DELETE) result ACCEPTED, REJECTED and (DELTA, DOCUMENT for update
	 * only).
	 *
	 * @param shadowName
	 *            the name of shadow
	 * @param action
	 *            the shadow action see {@link ShadowAction} for possible values
	 * @param result
	 *            the result to listen too ACCEPTED, REJECTED and (DELTA, DOCUMENT for update only). Use
	 *            {@link ShadowResult} for possible values
	 * @param callback
	 *            the action callback will be executed when a message is received.
	 * @param qos
	 *            the MQTT quality of service.
	 * @throws AwsIotException
	 *             on error.
	 */
	public void subscribeToShadow(String shadowName, ShadowAction action, ShadowResult result,
			AwsIotMessageCallback callback, int qos) throws AwsIotException {

		String topic = buildShadowTopicName(shadowName, action, result);
		subscribe(topic, callback, qos);
	}

	/**
	 * Unsubscribe from a shadow action.
	 *
	 *
	 * @param action
	 *            the shadow action see {@link ShadowAction} for possible values
	 * @param result
	 *            the result to listen too ACCEPTED, REJECTED and (DELTA, DOCUMENT for update only). Use
	 *            {@link ShadowResult} for possible values
	 * @throws AwsIotException
	 *             on error
	 */
	public void unsubscribeFromShadow(ShadowAction action, ShadowResult result) throws AwsIotException {

		String topic = buildShadowTopicName(null, action, result);
		unsubscribe(topic);
	}

	/**
	 * Unsubscribe from a shadow action.
	 *
	 * @param shadowName
	 *            shadow name
	 * @param action
	 *            the shadow action see {@link ShadowAction} for possible values
	 * @param result
	 *            the result to listen too ACCEPTED, REJECTED and (DELTA, DOCUMENT for update only). Use
	 *            {@link ShadowResult} for possible values
	 * @throws AwsIotException
	 *             on error
	 */
	public void unsubscribeFromShadow(String shadowName, ShadowAction action, ShadowResult result)
			throws AwsIotException {

		String topic = buildShadowTopicName(shadowName, action, result);
		unsubscribe(topic);
	}

	/**
	 * Gets the client options.
	 *
	 * @return the client options.
	 */
	public AwsIotClientOptions getClientOptions() {
		return this.options;
	}

	/**
	 * dispatch message to corresponding subscriber if any otherwise log a warning if warning logs are activated
	 *
	 * @param message
	 *            received message
	 */
	protected void dispatch(final AwsIotMessage message) {
		Map<String, AwsIotMessageCallback> subscribers = this.subscribers;
		assert subscribers != null;

		if (subscribers.containsKey(message.getTopic())) { // TODO support pattern matching?
			final AwsIotMessageCallback callback = subscribers.get(message.getTopic());
			assert callback != null;
			ExecutorService executorService = this.executorService;
			assert executorService != null;
			executorService.execute(new MessageDispatcher(callback, message));
		}
	}

	private static final class MessageDispatcher implements Runnable {

		private final AwsIotMessageCallback callback;
		private final AwsIotMessage message;

		public MessageDispatcher(AwsIotMessageCallback callback, AwsIotMessage message) {
			this.callback = callback;
			this.message = message;
		}

		@Override
		public void run() {
			AwsIotMessageCallback callback = this.callback;
			assert callback != null;
			AwsIotMessage message = this.message;
			assert message != null;
			callback.onMessageReceived(message);
		}
	}

	/**
	 * Dispatch connection lost event if an {@link AwsIotConnectionListener} was registered.
	 *
	 * @param cause
	 *            connection lost cause
	 */
	protected void connectionLost(final Throwable cause) {
		final AwsIotConnectionListener connectionListener = this.connectionListener;
		if (connectionListener != null) {
			connectionListener.connectionLost(cause);
		}
	}

	/**
	 * Build shadow topic from shadow name action and method.
	 *
	 * @param shadowName
	 *            shadow name. null for classic shadows.
	 * @param action
	 *            shadow action: GET, DELETE, UPDATE
	 * @param result
	 *            action result: accepted, rejected...
	 * @return shadow topic name
	 */
	private String buildShadowTopicName(String shadowName, ShadowAction action, ShadowResult result) {

		if ((ShadowAction.get.equals(action) || ShadowAction.delete.equals(action))
				&& (ShadowResult.delta.equals(result) || ShadowResult.documents.equals(result))) {
			throw new IllegalArgumentException();
		}

		final AwsIotClientOptions options = this.options;

		final StringBuilder topic = new StringBuilder();
		topic.append(SHADOW_BASE);
		topic.append(options.getThingName());
		topic.append(SHADOW);
		if (shadowName != null && !shadowName.isEmpty()) {
			topic.append(NAME).append(shadowName);
		}

		topic.append(SLASH).append(action.name());

		if (result != null) {
			topic.append(SLASH).append(result.name());
		}

		return topic.toString();
	}

}
