/*
 * Copyright 2016-2024 MicroEJ Corp. All rights reserved.
 * This library is provided in source code for use, modification and test, subject to license terms.
 * Any modification of the source code will break MicroEJ Corp. warranties on the whole library.
 */
package ej.mwt.stylesheet.selector;

import ej.annotation.Nullable;
import ej.mwt.Widget;

/**
 * A type or subtype selector selects by checking the widget type hierarchy (Java class).
 * <p>
 * Does not exists in CSS since there is no inheritance between tags. Its specificity is (0,0,0,1).
 *
 * @see Object#getClass()
 * @see Class#isAssignableFrom(Class)
 * @see SelectorHelper
 */
public class TypeSelector implements Selector {

	private final Class<? extends Widget> type;

	/**
	 * Creates a type or subtype selector.
	 *
	 * @param type
	 *            the widget subtype to check.
	 */
	public TypeSelector(Class<? extends Widget> type) {
		this.type = type;
	}

	@Override
	public boolean appliesToWidget(Widget widget) {
		return this.type.isAssignableFrom(widget.getClass());
	}

	@Override
	public int getSpecificity() {
		return SelectorHelper.getSpecificity(0, 0, 0, 1);
	}

	@Override
	public boolean equals(@Nullable Object obj) {
		if (obj != null && getClass() == obj.getClass()) {
			return ((TypeSelector) obj).type == this.type;
		}
		return false;
	}

	@Override
	public int hashCode() {
		return this.type.hashCode();
	}
}
