/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.net;

import java.util.Date;

/**
 * This class is used to process the bandwidth limit checking.
 *
 * @see #check(long, boolean)
 */
public class BandwidthChecker {
	private static final long DAY_TO_MS = (1000 * 3600 * 24);
	private static final long HOUR_TO_MS = (1000 * 3600);
	private static final long SECOND_TO_MS = 1000;

	private final long maxByteCount;
	private final Bandwidth.Period period;
	private long byteCount;
	private long startTime;
	private long timeIncrement;

	/**
	 * Creates an instance of the bandwidth checker.
	 * <p>
	 * If the given <code>maxBytes</code> is negative then there is no bandwidth limit.
	 *
	 * @param period
	 * 		the period over which the limit is applied.
	 * @param maxBytes
	 * 		the maximum number of bytes allowed.
	 */
	public BandwidthChecker(Bandwidth.Period period, long maxBytes) {
		this.startTime = System.currentTimeMillis();
		this.byteCount = 0;
		this.maxByteCount = maxBytes;
		this.period = period;
		switch (period) {
		case SECOND:
			this.timeIncrement = SECOND_TO_MS;
			break;
		case HOUR:
			this.timeIncrement = HOUR_TO_MS;
			break;
		case DAY:
			this.timeIncrement = DAY_TO_MS;
			break;
		default:
			break;
		}
	}

	/**
	 * Returns the maximum allowed number of bytes to read/write per period.
	 *
	 * @return the maximum allowed number of bytes read/written per period.
	 */
	public long getMaxByteCount() {
		return this.maxByteCount;
	}

	/**
	 * Returns the bandwidth verification period.
	 *
	 * @return the bandwidth verification period.
	 */
	public Bandwidth.Period getPeriod() {
		return this.period;
	}

	/**
	 * Check bandwidth limit. This method should be called before performing a read/write operation.
	 *
	 * @param nbBytes
	 * 		the number of bytes to be read/written. The check is skipped is this value is negative or null.
	 * @param write
	 * 		a boolean flag indicating whether it is a write operation.
	 * @return the number of bytes that can be read/written.
	 * @throws BandwidthLimitException
	 * 		if the daily bandwidth limit is exceeded.
	 */
	public synchronized long check(long nbBytes, boolean write) throws BandwidthLimitException {
		if ((nbBytes <= 0) || (this.maxByteCount < 0)) {
			return nbBytes;
		}

		long elapsedTime = System.currentTimeMillis() - this.startTime;

		// Check if the current elapsed time is greater than the verification period.
		if (elapsedTime > this.timeIncrement) {
			// Increment the start time when a full period has elapsed.
			this.startTime += this.timeIncrement;
			// Reset the total byte count.
			this.byteCount = 0;
		}

		long available = this.maxByteCount - this.byteCount;
		if (available <= 0) {
			// No bandwidth available until startTime.
			throw new BandwidthLimitException(new Date(this.startTime + this.timeIncrement));
		} else {
			if (write) {
				if (nbBytes > available) {
					// No bandwidth available until startTime.
					throw new BandwidthLimitException(new Date(this.startTime + this.timeIncrement));
				}
				return nbBytes;
			}
			else {
				if (available > nbBytes) {
					available = nbBytes;
				}
				return available;
			}
		}
	}

	/**
	 * Update bandwidth limit. This method should be called after having performed the read/write operation.
	 *
	 * @param nbBytes
	 * 		the number of bytes actually read/written. The update is skipped is this value is negative or null.
	 */
	public synchronized void update(long nbBytes) {
		this.byteCount += nbBytes;
	}

	@Override
	public String toString() {
		return "[Period (ms): " + this.timeIncrement + ", Data-limit (bytes): " + this.maxByteCount + "]";  // $NON-NLS-1$
	}
}
