/*
 * Java
 *
 * Copyright 2025 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.kf.util.control.fs;

import com.microej.kf.util.module.SandboxedModule;
import com.microej.kf.util.module.SandboxedModuleHelper;
import ej.service.ServiceFactory;

import java.io.File;
import java.io.RandomAccessFile;

/**
 * Action performed when a file is opened.
 */
public class OpenFile extends AbstractFile {

	private final boolean writeMode;
	private final boolean overwrite;
	private final long overwritingSize;

	/**
	 * Creates an open file action.
	 *
	 * @param file
	 * 		the file to be opened.
	 * @param writeMode
	 * 		true if the file is opened in write mode.
	 * @param overwrite
	 * 		true if the file will be overwritten if it exists; false otherwise. This parameter has no effect if
	 * 		<code>writeMode</code> is set to false.
	 */
	public OpenFile(File file, boolean writeMode, boolean overwrite) {
		super(file);
		this.writeMode = writeMode;
		this.overwrite = overwrite;
		this.overwritingSize = (overwrite) ? file.length() : 0;
	}

	/**
	 * Creates an open file action in read mode.
	 *
	 * @param file
	 * 		the file to be opened in read mode.
	 */
	public OpenFile(File file) {
		this(file, false, false);
	}

	public OpenFile(RandomAccessFile file) {
		this(null, false, false);
	}

	/**
	 * Tells whether the file will be overwritten if it exists.
	 *
	 * @return true if the file will be overwritten; false otherwise.
	 */
	public boolean isOverwritten() {
		return this.overwrite;
	}

	/**
	 * Tells whether the file is opened in write mode or not.
	 *
	 * @return true if the file is opened in write mode; false otherwise.
	 */
	public boolean isWriteMode() {
		return this.writeMode;
	}

	/**
	 * Returns the size of the file before overwriting.
	 *
	 * @return the size of the file before it's overwriting.
	 */
	public long getOverwritingSize() {
		return this.overwritingSize;
	}

	/**
	 * Called when the execution of the action is about to start.
	 *
	 * This hook is used to check if the action is allowed to be performed.
	 *
	 * @throws SecurityException
	 * 		if the action is not allowed to be performed.
	 */
	public void onStart() throws SecurityException {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getFileSystemResourceController().onStart(currentModule, this);
	}

	/**
	 * Called when the execution of the action is finished.
	 *
	 * This hook is used to update the current resources state of the module that performs the action.
	 *
	 * @param endWithSuccess
	 * 		true if the execution of the action was ended with success; false otherwise.
	 */
	public void onEnd(boolean endWithSuccess) {
		SandboxedModuleHelper cmh = ServiceFactory.getRequiredService(SandboxedModuleHelper.class);
		SandboxedModule currentModule = cmh.getCurrentModule();
		currentModule.getFileSystemResourceController().onEnd(currentModule, this, endWithSuccess);
	}
}
