/*
 * Copyright 2022 MicroEJ Corp. This file has been modified by MicroEJ Corp.
 *
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.regexp;

import java.io.IOException;
import java.io.Reader;

/**
 * Encapsulates java.io.Reader as CharacterIterator
 *
 * @author <a href="mailto:ales.novak@netbeans.com">Ales Novak</a>
 * @version CVS $Id: ReaderCharacterIterator.java 518156 2007-03-14 14:31:26Z vgritsenko $
 */
public final class ReaderCharacterIterator implements CharacterIterator {
	/** Underlying reader */
	private final Reader reader;

	/** Buffer of read chars */
	private final StringBuffer buff;

	/** read end? */
	private boolean closed;

	/**
	 * @param reader
	 *            a Reader, which is parsed
	 */
	public ReaderCharacterIterator(Reader reader) {
		this.reader = reader;
		this.buff = new StringBuffer(512);
		this.closed = false;
	}

	/** @return a substring */
	@Override
	public String substring(int beginIndex, int endIndex) {
		try {
			ensure(endIndex);
			return this.buff.toString().substring(beginIndex, endIndex);
		} catch (IOException e) {
			String message = e.getMessage();
			if (message == null) {
				throw new StringIndexOutOfBoundsException();
			} else {
				throw new StringIndexOutOfBoundsException(message);
			}
		}
	}

	/** @return a substring */
	@Override
	public String substring(int beginIndex) {
		try {
			readAll();
			return this.buff.toString().substring(beginIndex);
		} catch (IOException e) {
			String message = e.getMessage();
			if (message == null) {
				throw new StringIndexOutOfBoundsException();
			} else {
				throw new StringIndexOutOfBoundsException(message);
			}
		}
	}

	/** @return a character at the specified position. */
	@Override
	public char charAt(int pos) {
		try {
			ensure(pos);
			return this.buff.charAt(pos);
		} catch (IOException e) {
			String message = e.getMessage();
			if (message == null) {
				throw new StringIndexOutOfBoundsException();
			} else {
				throw new StringIndexOutOfBoundsException(message);
			}
		}
	}

	/** @return <tt>true</tt> iff if the specified index is after the end of the character stream */
	@Override
	public boolean isEnd(int pos) {
		if (this.buff.length() > pos) {
			return false;
		} else {
			try {
				ensure(pos);
				return (this.buff.length() <= pos);
			} catch (IOException e) {
				String message = e.getMessage();
				if (message == null) {
					throw new StringIndexOutOfBoundsException();
				} else {
					throw new StringIndexOutOfBoundsException(message);
				}
			}
		}
	}

	/** Reads n characters from the stream and appends them to the buffer */
	private int read(int n) throws IOException {
		if (this.closed) {
			return 0;
		}

		char[] c = new char[n];
		int count = 0;
		int read = 0;

		do {
			read = this.reader.read(c);
			if (read < 0) // EOF
			{
				this.closed = true;
				break;
			}
			count += read;
			this.buff.append(c, 0, read);
		} while (count < n);

		return count;
	}

	/** Reads rest of the stream. */
	private void readAll() throws IOException {
		while (!this.closed) {
			read(1000);
		}
	}

	/** Reads chars up to the idx */
	private void ensure(int idx) throws IOException {
		if (this.closed) {
			return;
		}

		if (idx < this.buff.length()) {
			return;
		}
		read(idx + 1 - this.buff.length());
	}
}
