/*
 * Java
 *
 * Copyright 2015-2016 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.io.progress;

import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;
import org.eclipse.core.runtime.SubMonitor;

/**
 * Wraps an input stream to follow its reading progression.
 */
public class ProgressInputStream extends FilterInputStream {

	private final IProgressMonitor monitor;

	/**
	 * Creates a progress input stream with a listener specifying the total number of bytes to read.
	 *
	 * @param in
	 *            the underlying input stream.
	 * @param monitor
	 *            the listener to notify or <code>null</code>.
	 * @param totalBytes
	 *            the total number of bytes to be read.
	 */
	public ProgressInputStream(InputStream in, int totalBytes, IProgressMonitor monitor){
		super(in);
		if (totalBytes > 0) {
			this.monitor = SubMonitor.convert(monitor, totalBytes);
		} else {
			this.monitor = new NullProgressMonitor();
		}
	}

	@Override
	public int read() throws IOException {
		int read = super.read();
		if (read != -1) {
			progress(1);
		}
		return read;
	}

	@Override
	public int read(byte[] b, int off, int len) throws IOException {
		int read = super.read(b, off, len);
		progress(read);
		return read;
	}

	@Override
	public long skip(long n) throws IOException {
		long skip = super.skip(n);
		progress((int) skip);
		return skip;
	}

	@Override
	public synchronized void mark(int readlimit) {
		throw new UnsupportedOperationException();
	}

	@Override
	public synchronized void reset() throws IOException {
		throw new UnsupportedOperationException();
	}

	@Override
	public boolean markSupported() {
		return false;
	}

	private void progress(int read) {
		if (read != -1) {
			this.monitor.worked(read);
		}
	}
}
