/*
 * Java
 *
 * Copyright 2010-2017 IS2T. All rights reserved.
 * IS2T PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package ej.microui.display;

import java.io.IOException;
import java.io.InputStream;

/**
 * An <code>Image</code> object holds graphical display data. <br>
 * An <code>Image</code> is either mutable or immutable depending on the way it has been created.<br>
 * <br>
 * Immutable images are created/loaded from data resources, and they may not be modified. They can be either loaded from
 * an internal non-standard image format or dynamically created from supported standard image formats, depending on the
 * MicroUI implementation.<br>
 * <br>
 * Creating images may not be possible within a particular MicroUI implementation. If the MicroUI implementation doesn't
 * allow the allocation of memory to store the image's buffer, an {@link OutOfMemoryError} is thrown.<br>
 * <br>
 * Mutable images are created as blank images containing only white pixels. The application may render on a mutable
 * image by calling {@link #getGraphicsContext()} on the image to obtain a {@link GraphicsContext} object expressly for
 * this purpose.
 */
public class Image {

	/**
	 * Specify the format to apply when creating an immutable image.
	 *
	 * @see #createImage(String, OutputFormat)
	 * @see #createImage(InputStream, int, OutputFormat)
	 * @see #createImage(Image, int, int, int, int, OutputFormat)
	 * @since 2.0
	 */
	public enum OutputFormat {
		/**
		 * Each pixel value is stored on 32 bits, in <code>0xARGB</code> format. <br>
		 * 8 bits for <code>alpha</code> (0 to 255)<br>
		 * 8 bits for <code>red</code> (0 to 255)<br>
		 * 8 bits for <code>green</code> (0 to 255)<br>
		 * 8 bits for <code>blue</code> (0 to 255)<br>
		 */
		ARGB8888(2),

		/**
		 * Each pixel value is stored on 24 bits, in <code>0xRGB</code> format<br>
		 * 0 bits for <code>alpha</code><br>
		 * 8 bits for <code>red</code> (0 to 255)<br>
		 * 8 bits for <code>green</code> (0 to 255)<br>
		 * 8 bits for <code>blue</code> (0 to 255)<br>
		 */
		RGB888(3),

		/**
		 * Each pixel value is stored on 16 bits, in <code>0xRGB</code> format<br>
		 * 0 bits for <code>alpha</code> (0 to 255)<br>
		 * 5 bits for <code>red</code> (0 to 31)<br>
		 * 6 bits for <code>green</code> (0 to 63)<br>
		 * 5 bits for <code>blue</code> (0 to 31)<br>
		 */
		RGB565(4),

		/**
		 * Each pixel value is stored on 16 bits, in <code>0xARGB</code> format<br>
		 * 1 bits for <code>alpha</code> (0 to 1)<br>
		 * 5 bits for <code>red</code> (0 to 31)<br>
		 * 5 bits for <code>green</code> (0 to 31)<br>
		 * 5 bits for <code>blue</code> (0 to 31)<br>
		 */
		ARGB1555(5),

		/**
		 * Each pixel value is stored on 16 bits, in <code>0xARGB</code> format<br>
		 * 4 bits for <code>alpha</code> (0 to 15)<br>
		 * 4 bits for <code>red</code> (0 to 15)<br>
		 * 4 bits for <code>green</code> (0 to 15)<br>
		 * 4 bits for <code>blue</code> (0 to 15)<br>
		 */
		ARGB4444(6),

		/**
		 * Each pixel value is stored on 8 bits, in <code>0xA</code> format<br>
		 * 8 bits for <code>alpha</code> (0 to 255)<br>
		 * 0 bits for <code>red</code><br>
		 * 0 bits for <code>green</code><br>
		 * 0 bits for <code>blue</code><br>
		 */
		A8(8);

		private OutputFormat(int val) {
			throw new RuntimeException();
		}
	}

	/**
	 * Forbidden constructor: use <code>createImage(...)</code> methods to get an instance of {@link Image}.
	 *
	 * @see #createImage(int, int)
	 * @see #createImage(String, OutputFormat)
	 * @see #createImage(InputStream, int, OutputFormat)
	 * @see #createImage(Image, int, int, int, int, OutputFormat)
	 */
	private Image() {
	}

	/**
	 * Creates a new mutable image which respects the pixel organization (layout, bpp etc.) of the default display and
	 * with the given size. Every pixel within the newly created image is white. Given <code>width</code> and
	 * <code>height</code> must be greater than zero.The effect is identical to<br>
	 * <code>createImage(Display.getDefaultDisplay(), width, height);</code>
	 *
	 * @param width
	 *            the width of the new image, in pixels.
	 * @param height
	 *            the height of the new image, in pixels.
	 * @return the created image.
	 * @throws IllegalArgumentException
	 *             if either <code>width</code> or <code>height</code> is zero or less.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @see #createImage(Display, int, int)
	 */
	public static Image createImage(int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Creates a new mutable image which respects the pixel organization (layout, bpp etc.) of the given display and
	 * with the given size. Every pixel within the newly created image is white. Given <code>width</code> and
	 * <code>height</code> must be greater than zero.<br>
	 *
	 * @param d
	 *            the display for which the image is created.
	 * @param width
	 *            the width of the new image, in pixels.
	 * @param height
	 *            the height of the new image, in pixels.
	 * @return the created image.
	 * @throws IllegalArgumentException
	 *             if either <code>width</code> or <code>height</code> is zero or less
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 */
	public static Image createImage(Display d, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Creates an immutable image from a resource.<br>
	 * <br>
	 * The resource image format (input format) is a standard image format (<code>PNG</code> etc.) or an internal
	 * MicroUI implementation specific format. According the MicroUI implementation the image can be loaded or not. For
	 * instance, when MicroUI implementation does not hold a dynamic <code>PNG</code> decoder, load a <code>PNG</code>
	 * is impossible. <br>
	 * The MicroUI implementation is responsible to retrieve the right image decoder and/or loader to decode and/or load
	 * the image. <br>
	 * When a resource cannot be loaded, an {@link IOException} is thrown. <br>
	 * <br>
	 * The output format is the best format for the input format. For instance the <code>PNG</code> image output format
	 * is {@link OutputFormat#ARGB8888} (32 bits per pixels with 256 levels of transparency). <br>
	 * The output format for the MicroUI implementation specific images formats is often the encoded format.<br>
	 * <br>
	 * The effect is identical to<br>
	 * <code>createImage(name, null);</code>
	 *
	 * @param name
	 *            a resource name matching image data in a supported image format.
	 * @return the created image.
	 * @throws NullPointerException
	 *             if name is <code>null</code>.
	 * @throws IOException
	 *             if the resource is not found, if the data can not be decoded or if the name is not an absolute path.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 * @see #createImage(String, OutputFormat)
	 * @since 2.0
	 */
	public static Image createImage(String name) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Creates an immutable image from a resource.<br>
	 * <br>
	 * The resource image format (input format) is a standard image format (<code>PNG</code> etc.) or an internal
	 * MicroUI implementation specific format. According the MicroUI implementation the image can be loaded or not. For
	 * instance, when MicroUI implementation does not hold a dynamic <code>PNG</code> decoder, load a <code>PNG</code>
	 * is impossible. <br>
	 * The MicroUI implementation is responsible to retrieve the right image decoder and/or loader to decode and/or load
	 * the image. <br>
	 * When a resource cannot be loaded, an {@link IOException} is thrown. <br>
	 * <br>
	 * The output format is the format specified as paremeter. This format may be different than the best format for the
	 * input format (@see {@link #createImage(String)}.<br>
	 *
	 * @param name
	 *            a resource name matching image data in a supported image format.
	 * @param format
	 *            the expected output format or <code>null</code> to use the best output format according the input
	 *            image format.
	 * @return the created image.
	 * @throws NullPointerException
	 *             if name is <code>null</code>.
	 * @throws IOException
	 *             if the resource is not found, if the data can not be decoded or if the name is not an absolute path.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 * @since 2.0
	 */
	public static Image createImage(String name, OutputFormat format) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Creates an immutable image from another image area. The image is created with the same format as the original
	 * image.<br>
	 * The area of the source image to copy is defined by an upper-left location <code>(x,y)</code> relative to the
	 * image and a size <code>(width,height)</code>.<br>
	 * <br>
	 * If the defined zone matches the entire source image and if the source image is immutable, then the source image
	 * <code>image</code> may be returned. <br>
	 * <br>
	 * The effect is identical to<br>
	 * <code>createImage(image, x, y, width, height, null);</code>
	 *
	 * @param image
	 *            the source image.
	 * @param x
	 *            the x coordinate of the area to copy.
	 * @param y
	 *            the y coordinate of the area to copy.
	 * @param width
	 *            the width of the area to copy.
	 * @param height
	 *            the height of the area to copy.
	 * @return an immutable image.
	 * @throws NullPointerException
	 *             if image is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the area to copy is out of the bounds of the source image or if either <code>width</code> or
	 *             <code>height</code> is zero or negative.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @see #createImage(Image, int, int, int, int, OutputFormat)
	 */
	public static Image createImage(Image image, int x, int y, int width, int height) {
		throw new RuntimeException();
	}

	/**
	 * Creates an immutable image from another image area. The image is created with the specified format.<br>
	 * The area of the source image to copy is defined by an upper-left location <code>(x,y)</code> relative to the
	 * image and a size <code>(width,height)</code>.<br>
	 * <br>
	 * If the defined zone matches the entire source image and if the source image is immutable, then the source image
	 * <code>image</code> may be returned.
	 *
	 * @param image
	 *            the source image.
	 * @param x
	 *            the x coordinate of the area to copy.
	 * @param y
	 *            the y coordinate of the area to copy.
	 * @param width
	 *            the width of the area to copy.
	 * @param height
	 *            the height of the area to copy.
	 * @param format
	 *            the expected output format or <code>null</code> to use the source image format.
	 * @return an immutable image
	 * @throws NullPointerException
	 *             if image is <code>null</code>.
	 * @throws IllegalArgumentException
	 *             if the area to copy is out of the bounds of the source image or if either <code>width</code> or
	 *             <code>height</code> is zero or negative.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @since 2.0
	 */
	public static Image createImage(Image image, int x, int y, int width, int height, OutputFormat format) {
		throw new RuntimeException();
	}

	/**
	 * Creates an immutable image from an {@link InputStream}.<br>
	 * <br>
	 * The resource image format (input format) is a standard image format (<code>PNG</code> etc.) According the MicroUI
	 * implementation the image can be loaded or not. For instance, when MicroUI implementation does not hold a dynamic
	 * <code>PNG</code> decoder, load a <code>PNG</code> is impossible. <br>
	 * The MicroUI implementation is responsible to retrieve the right image decoder and/or loader to decode and/or load
	 * the image. <br>
	 * When a resource cannot be loaded, an {@link IOException} is thrown. <br>
	 * <br>
	 * The output format is the best format for the input format. For instance the <code>PNG</code> image output format
	 * is {@link OutputFormat#ARGB8888} (32 bits per pixels with 256 levels of transparency). <br>
	 * <br>
	 * The effect is identical to<br>
	 * <code>createImage(stream, size, format);</code>
	 *
	 * @param stream
	 *            a stream providing image data.
	 * @param size
	 *            the number of bytes to read.
	 * @return the created image
	 * @throws NullPointerException
	 *             if stream is <code>null</code>.
	 * @throws IOException
	 *             if the resource is not found, if the data can not be decoded or if the stream cannot be read.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 * @see #createImage(InputStream, int, OutputFormat)
	 * @since 2.0
	 */
	public static Image createImage(InputStream stream, int size) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Creates an immutable image from an {@link InputStream}.<br>
	 * <br>
	 * The resource image format (input format) is a standard image format ( <code>PNG</code> etc.) According the
	 * MicroUI implementation the image can be loaded or not. For instance, when MicroUI implementation does not hold a
	 * dynamic <code>PNG</code> decoder, load a <code>PNG</code> is impossible. <br>
	 * The MicroUI implementation is responsible to retrieve the right image decoder and/or loader to decode and/or load
	 * the image. <br>
	 * When a resource cannot be loaded, an {@link IOException} is thrown. <br>
	 * <br>
	 * The output format is the format specified as paremeter. This format may be different than the best format for the
	 * input format (@see {@link #createImage(InputStream, int)}.<br>
	 *
	 * @param stream
	 *            a stream providing image data
	 * @param size
	 *            the number of bytes to read
	 * @param format
	 *            the expected output format or null to use the best output format according the input image format
	 * @return the created image
	 * @throws NullPointerException
	 *             if stream is null
	 * @throws IOException
	 *             if the resource is not found, if the data can not be decoded or if the stream cannot be read.
	 * @throws ImageCreationException
	 *             if MicroUI implementation cannot create the image
	 * @throws SecurityException
	 *             if a security manager exists and does not allow the caller to create an image.
	 * @throws OutOfMemoryError
	 *             if there is not enough room to add a new image.
	 * @throws IllegalStateException
	 *             if MicroUI is not started.
	 * @since 2.0
	 */
	public static Image createImage(InputStream stream, int size, OutputFormat format) throws IOException {
		throw new RuntimeException();
	}

	/**
	 * Returns a new {@link GraphicsContext} object to draw on the image. The image must be mutable, otherwise an
	 * {@link IllegalArgumentException} is thrown. The returned {@link GraphicsContext} object has the default
	 * {@link GraphicsContext} behavior and allows the entire image to be drawn.
	 *
	 * @return a {@link GraphicsContext} object which maps this image
	 * @throws IllegalArgumentException
	 *             if the image is immutable
	 */
	public GraphicsContext getGraphicsContext() {
		throw new RuntimeException();
	}

	/**
	 * Returns the width of the image in pixels.
	 *
	 * @return width of the image
	 */
	public int getWidth() {
		throw new RuntimeException();
	}

	/**
	 * Returns the height of the image in pixels.
	 *
	 * @return height of the image
	 *
	 */
	public int getHeight() {
		throw new RuntimeException();
	}

	/**
	 * Tells whether this image is mutable.
	 *
	 * @return whether the image is mutable.
	 */
	public boolean isMutable() {
		throw new RuntimeException();
	}

	/**
	 * Obtains the ARGB color of the pixel at <code>(x,y)</code>. The read color may be different than the drawing
	 * color. It is screen dependent, according to the number of bits per pixels (see {@link Display#getBPP()} ).
	 *
	 * @param x
	 *            the x coordinate of the pixel
	 * @param y
	 *            the y coordinate of the pixel
	 * @return the ARGB color of the pixel
	 * @throws IllegalArgumentException
	 *             if the pixel coordinates are out of bounds of the source graphics context
	 */
	public int readPixel(int x, int y) {
		throw new RuntimeException();
	}

	/**
	 * Obtains ARGB pixel data from the specified region of this image and stores it in the provided array of integers.
	 * <br>
	 * Each pixel value is stored in 0xAARRGGBB format, where the high-order byte contains the alpha channel and the
	 * remaining bytes contain color components for red, green and blue, respectively. The alpha channel specifies the
	 * opacity of the pixel, where a value of 0x00 represents a pixel that is fully transparent and a value of 0xFF
	 * represents a fully opaque pixel.<br>
	 * <p>
	 * The returned values are not guaranteed to be identical to values from the original source, such as from
	 * createRGBImage or from a PNG image. Color values may be resampled to reflect the display capabilities of the
	 * device (for example, red, green or blue pixels may all be represented by the same gray value on a grayscale
	 * device). On devices that do not support alpha blending, the alpha value will be 0xFF for opaque pixels and 0x00
	 * for all other pixels. On devices that support alpha blending, alpha channel values may be resampled to reflect
	 * the number of levels of semitransparency supported.<br>
	 * <p>
	 * The scanlength specifies the relative offset within the array between the corresponding pixels of consecutive
	 * rows. In order to prevent rows of stored pixels from overlapping, the absolute value of scanlength must be
	 * greater than or equal to width. Negative values of scanlength are allowed. In all cases, this must result in
	 * every reference being within the bounds of the rgbData array.<br>
	 *
	 * @param argbData
	 *            an array of integers in which the ARGB pixel data is stored
	 * @param offset
	 *            the index into the array where the first ARGB value is stored
	 * @param scanlength
	 *            the relative offset in the array between corresponding pixels in consecutive rows of the region
	 * @param x
	 *            the x-coordinate of the upper left corner of the region
	 * @param y
	 *            the y-coordinate of the upper left corner of the region
	 * @param width
	 *            the width of the region
	 * @param height
	 *            the height of the region
	 * @throws ArrayIndexOutOfBoundsException
	 *             if the requested operation would attempt to access an element in the rgbData array whose index is
	 *             either negative or beyond its length (the contents of the array are unchanged)
	 * @throws IllegalArgumentException
	 *             if the area being retrieved exceeds the bounds of the source image
	 * @throws IllegalArgumentException
	 *             if the absolute value of scanlength is less than width
	 * @throws NullPointerException
	 *             - if argbData is null
	 */
	public void getARGB(int[] argbData, int offset, int scanlength, int x, int y, int width, int height) {
		throw new RuntimeException();
	}
}