/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microvg.Matrix;
import ej.microvg.ResourceVectorImage;
import ej.microvg.VectorGraphicsException;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorImage;

/**
 * Test the loading and the use of external fonts.
 */
@SuppressWarnings("nls")
public class TestImageExternal {

	private static final String INTERNAL_IMAGE_REFERENCE = "/com/microej/microvg/test/external_reference.xml";
	private static final String EXTERNAL_IMAGE_ADDRESSABLE_VALID = "/com/microej/microvg/test/external_addr_valid.xml";
	private static final String EXTERNAL_IMAGE_ADDRESSABLE_INVALID = "/com/microej/microvg/test/external_addr_invalid.xml";
	private static final String EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID = "/com/microej/microvg/test/external_naddr_valid.xml";
	private static final String EXTERNAL_IMAGE_NOT_ADDRESSABLE_INVALID = "/com/microej/microvg/test/external_naddr_invalid.xml";

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Tests loading an image that exists in the external memory and byte addressable.
	 */
	@Test
	public void testExistingExternalImageByteAddressable() {
		VectorImage ref = VectorImage.getImage(INTERNAL_IMAGE_REFERENCE);

		try (ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID)) {
			Assert.assertEquals(ref.getHeight(), image.getHeight(), 0.0f);
			Assert.assertEquals(ref.getWidth(), image.getWidth(), 0.0f);
			Assert.assertEquals(ref.getDuration(), image.getDuration(), 0.0f);
		}

	}

	/**
	 * Tests loading an image from non-byte-addressable external memory.
	 */
	@Test
	public void testExistingExternalFontNotByteAddressable() {
		VectorImage ref = VectorImage.getImage(INTERNAL_IMAGE_REFERENCE);

		try (ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID)) {
			Assert.assertEquals(ref.getHeight(), image.getHeight(), 0.0f);
			Assert.assertEquals(ref.getWidth(), image.getWidth(), 0.0f);
			Assert.assertEquals(ref.getDuration(), image.getDuration(), 0.0f);
		}
	}

	/**
	 * Tests loading twice an image that exists in the external memory and byte addressable.
	 */
	@Test
	public void testExistingExternalImageByteAddressableTwice() {
		try (ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID);
				ResourceVectorImage image2 = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID)) {
			Assert.assertEquals(image.getHeight(), image2.getHeight(), 0.0f);
			Assert.assertEquals(image.getWidth(), image2.getWidth(), 0.0f);
			Assert.assertEquals(image.getDuration(), image2.getDuration(), 0.0f);
		}
	}

	/**
	 * Tests loading twice an image from non-byte-addressable external memory.
	 */
	@Test
	public void testExistingExternalImageNotByteAddressableTwice() {
		try (ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID);
				ResourceVectorImage image2 = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID)) {
			Assert.assertEquals(image.getHeight(), image2.getHeight(), 0.0f);
			Assert.assertEquals(image.getWidth(), image2.getWidth(), 0.0f);
			Assert.assertEquals(image.getDuration(), image2.getDuration(), 0.0f);
		}
	}

	/**
	 * Tests closing an image loaded from non-byte-addressable external memory.
	 */
	@Test
	@SuppressWarnings("resource")
	public void testClosingNonByteAddressableExternalImage() {
		ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID);
		image.close();
	}

	/**
	 * Tests closing an image loaded from byte-addressable external memory.
	 */
	@Test()
	@SuppressWarnings("resource")
	public void testClosingByteAddressableExternalImage() {
		ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID);
		image.close();
	}

	/**
	 * Tests closing twice an image loaded from non-byte-addressable external memory.
	 */
	@Test()
	@SuppressWarnings("resource")
	public void testClosingNonByteAddressableExternalImageTwice() {
		ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID);
		image.close();
		image.close();
	}

	/**
	 * Tests using {@link VectorImage#getImage(String)} on an addressable external image: it must work because no
	 * allocation is required.
	 */
	@Test()
	public void testGetImageOnByteAddressableExternalImage() {
		VectorImage ref = VectorImage.getImage(INTERNAL_IMAGE_REFERENCE);
		VectorImage image = VectorImage.getImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID);

		Assert.assertEquals(ref.getHeight(), image.getHeight(), 0.0f);
		Assert.assertEquals(ref.getWidth(), image.getWidth(), 0.0f);
		Assert.assertEquals(ref.getDuration(), image.getDuration(), 0.0f);
	}

	/**
	 * Tests using {@link VectorImage#getImage(String)} on a non-addressable external image.
	 */
	@Test()
	public void testGetImageOnNonByteAddressableExternalImage() {
		try {
			VectorImage.getImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID);
			Assert.fail("exception is expected");
		} catch (VectorGraphicsException e) {
			Assert.assertEquals(e.getErrorCode(), VectorGraphicsException.IMAGE_INVALID_PATH);
		}
	}

	/**
	 * Tests drawing an addressable external image.
	 */
	@Test()
	public static void testDrawByteAddressableExternalImage() {
		Display display = Display.getDisplay();
		GraphicsContext graphicsContext = display.getGraphicsContext();

		try (ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID)) {
			VectorGraphicsPainter.drawImage(graphicsContext, image, new Matrix());

			display.flush();

			TestUtilities.check("Top side", 100, 50, graphicsContext, Colors.RED);
			TestUtilities.check("Bottom side", 100, 149, graphicsContext, Colors.RED);
			TestUtilities.check("Left side", 50, 100, graphicsContext, Colors.RED);
			TestUtilities.check("Right side", 149, 100, graphicsContext, Colors.RED);
			TestUtilities.check("Center", 100, 100, graphicsContext, Colors.RED);
		}
	}

	/**
	 * Tests drawing a non-addressable external image.
	 */
	@Test()
	public static void testDrawNonByteAddressableExternalImage() {
		Display display = Display.getDisplay();
		GraphicsContext graphicsContext = display.getGraphicsContext();

		try (ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID)) {
			VectorGraphicsPainter.drawImage(graphicsContext, image, new Matrix());

			display.flush();

			TestUtilities.check("Top side", 100, 50, graphicsContext, Colors.RED);
			TestUtilities.check("Bottom side", 100, 149, graphicsContext, Colors.RED);
			TestUtilities.check("Left side", 50, 100, graphicsContext, Colors.RED);
			TestUtilities.check("Right side", 149, 100, graphicsContext, Colors.RED);
			TestUtilities.check("Center", 100, 100, graphicsContext, Colors.RED);
		}
	}

	/**
	 * An internal image cannot be closed: the same image can be opened several times.
	 */
	@Test()
	public static void testMultipleOpenCloseInternalImage() {
		for (int i = 1000; --i >= 0;) {
			VectorImage.getImage(INTERNAL_IMAGE_REFERENCE);
		}
	}

	/**
	 * An external image must be closed: open/close the same image several times must work
	 */
	@Test()
	@SuppressWarnings("resource")
	public static void testMultipleOpenCloseByteAddressableExternalImage() {
		for (int i = 1000; --i >= 0;) {
			ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_VALID);
			image.close(); // useless actually
		}
	}

	/**
	 * An external image must be closed: open/close the same image several times must work
	 */
	@Test()
	@SuppressWarnings("resource")
	public static void testMultipleOpenCloseNonByteAddressableExternalImage() {
		for (int i = 1000; --i >= 0;) {
			ResourceVectorImage image = ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_VALID);
			image.close();
		}
	}

	/**
	 * An external image (byte addressable) can be invalid: an error code must be thrown
	 *
	 * For this test, the invalid image is generated by setting the first byte of the valid image to 0.
	 */
	@Test()
	public static void testGetInvalidByteAddressableExternalImage() {
		try {
			VectorImage.getImage(EXTERNAL_IMAGE_ADDRESSABLE_INVALID);
			Assert.fail("exception is expected");
		} catch (VectorGraphicsException e) {
			Assert.assertEquals(e.getErrorCode(), VectorGraphicsException.IMAGE_INVALID);
		}
	}

	/**
	 * An external image (byte addressable) can be invalid: an error code must be thrown
	 *
	 * For this test, the invalid image is generated by setting the first byte of the valid image to 0.
	 */
	@Test()
	public static void testLoadInvalidByteAddressableExternalImage() {
		try {
			ResourceVectorImage.loadImage(EXTERNAL_IMAGE_ADDRESSABLE_INVALID);
			Assert.fail("exception is expected");
		} catch (VectorGraphicsException e) {
			Assert.assertEquals(e.getErrorCode(), VectorGraphicsException.IMAGE_INVALID);
		}
	}

	/**
	 * An external image (no byte addressable) can be invalid: an error code must be thrown
	 *
	 * For this test, the invalid image is generated by setting the first byte of the valid image to 0.
	 */
	@Test()
	public static void testInvalidNonByteAddressableExternalImage() {
		try {
			ResourceVectorImage.loadImage(EXTERNAL_IMAGE_NOT_ADDRESSABLE_INVALID);
			Assert.fail("exception is expected");
		} catch (VectorGraphicsException e) {
			Assert.assertEquals(e.getErrorCode(), VectorGraphicsException.IMAGE_INVALID);
		}
	}

	/**
	 * A filtered image must be closed: open/close the same image several times must work
	 */
	@Test()
	@SuppressWarnings("resource")
	public static void testMultipleOpenCloseFilteredImage() {
		VectorImage ref = VectorImage.getImage(INTERNAL_IMAGE_REFERENCE);
		float[] colorMatrix = new float[] { //
				0.5f, 0, 0, 0, 0, // red
				0, 0, 0, 0, 0, // green
				0.5f, 0, 0, 0, 0, // blue
				0, 0, 0, 1, 0, // alpha
		};
		for (int i = 1000; --i >= 0;) {
			ResourceVectorImage image = ref.filterImage(colorMatrix);
			image.close();
		}
	}
}
