/*
 * Java
 *
 * Copyright 2022-2024 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package com.microej.microvg.test;

import org.junit.AfterClass;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import ej.microui.display.Colors;
import ej.microui.display.Display;
import ej.microui.display.GraphicsContext;
import ej.microui.display.Painter;
import ej.microvg.BlendMode;
import ej.microvg.BufferedVectorImage;
import ej.microvg.LinearGradient;
import ej.microvg.Matrix;
import ej.microvg.Path;
import ej.microvg.VectorFont;
import ej.microvg.VectorGraphicsPainter;
import ej.microvg.VectorGraphicsPainter.FillType;
import ej.microvg.VectorImage;

/**
 * Tests the drawing of MicroVG shapes into a BufferedVectorImage.
 */
@SuppressWarnings("nls")
public class TestBufferedVectorImageDraw {

	interface Drawer {

		/**
		 * Draws something in graphics context in the specified rectangle.
		 *
		 * @param g
		 *            the destination
		 * @param x
		 *            the rectangle top left X-coordinate
		 * @param y
		 *            the rectangle top left Y-coordinate
		 * @param w
		 *            the rectangle's width
		 * @param h
		 *            the rectangle's height
		 */
		void draw(GraphicsContext g, int x, int y, int w, int h);

	}

	/**
	 * Starts MicroUI.
	 */
	@BeforeClass
	public static void pre() {
		TestUtilities.startMicroUI();
	}

	/**
	 * Stops MicroUI.
	 */
	@AfterClass
	public static void post() {
		TestUtilities.stopMicroUI();
	}

	/**
	 * Resets the content of the screen to black.
	 */
	@Before
	public static void preTest() {
		Display display = Display.getDisplay();
		GraphicsContext g = display.getGraphicsContext();
		g.reset();
		g.setColor(TestUtilities.BACKGROUND_COLOR);
		Painter.fillRectangle(g, 0, 0, display.getWidth(), display.getHeight());
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: FillPath with translated matrix
	 */
	@Test
	public static void testVGPainterFillPathTranslatedMatrix() {
		testDrawer("FillPath", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.setColor(Colors.WHITE);
				VectorGraphicsPainter.fillPath(g, getPath(w, h), getTranslatedMatrix(x, y));
				g.setColor(Colors.RED);
				VectorGraphicsPainter.fillPath(g, getPath(w, h), getTranslatedMatrix(x, y), FillType.WINDING, 0x80,
						BlendMode.SRC_OVER);
			}
		}, 20, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: FillPath with gc translate
	 */
	@Test
	public static void testVGPainterFillPathTranslatedGC() {
		testDrawer("FillPath", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.translate(x, y);
				g.setColor(Colors.WHITE);
				VectorGraphicsPainter.fillPath(g, getPath(w, h), new Matrix());
				g.setColor(Colors.RED);
				VectorGraphicsPainter.fillPath(g, getPath(w, h), new Matrix(), FillType.WINDING, 0x80,
						BlendMode.SRC_OVER);
				g.translate(-x, -y);
			}
		}, 20, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: fillGradientPath with translated matrix
	 */
	@Test
	public static void testVGPainterFillGradientPathTranslatedMatrix() {
		testDrawer("fillGradientPath", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				LinearGradient gradient = new LinearGradient(0, 0, w - 1, h - 1,
						new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
				Matrix matrix = getTranslatedMatrix(x, y);
				VectorGraphicsPainter.fillGradientPath(g, getPath(w, h), matrix, gradient);
			}
		}, 20, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: fillGradientPath with translated gc
	 */
	@Test
	public static void testVGPainterFillGradientPathTranslatedGC() {
		testDrawer("fillGradientPath", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				LinearGradient gradient = new LinearGradient(0, 0, w - 1, h - 1,
						new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
				g.translate(x, y);
				VectorGraphicsPainter.fillGradientPath(g, getPath(w, h), new Matrix(), gradient);
				g.translate(-x, -y);
			}
		}, 20, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawString with translated gc
	 */
	@Test
	public static void testVGPainterDrawStringXY() {
		testDrawer("drawString", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.translate(x, y);
				g.setClip(0, 0, w, h);
				g.setColor(Colors.WHITE);
				VectorGraphicsPainter.drawString(g, "ABC", getFont(), h, 0, 0);
			}
		}, 50, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawString with translated matrix
	 */
	@Test
	public static void testVGPainterDrawStringMatrix() {
		testDrawer("drawString", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.setClip(x, y, w, h);
				g.setColor(Colors.WHITE);
				Matrix m = new Matrix();
				m.preTranslate(x, y);
				VectorGraphicsPainter.drawString(g, "ABC", getFont(), h, m, 0xff, BlendMode.SRC_OVER, 0);
			}
		}, 50, 20);
	}

	/**
	 * Tests some {@link Painter} algorithms: drawGradientString with translated gc
	 */
	@Test
	public static void testVGPainterDrawGradientStringTranslatedGC() {
		testDrawer("drawGradientString", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				LinearGradient gradient = new LinearGradient(0, 0, 99, 0,
						new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
				g.setClip(x, y, w, h);
				g.setColor(Colors.WHITE);
				g.translate(x, y);
				VectorGraphicsPainter.drawGradientString(g, "ABC", getFont(), h, new Matrix(), gradient,
						0xff, BlendMode.SRC_OVER, 0);
				g.translate(-x, -y);
			}
		}, 50, 20);
	}

	/**
	 * Tests some {@link Painter} algorithms: drawGradientString with translated matrix
	 */
	@Test
	public static void testVGPainterDrawGradientStringMatrix() {
		testDrawer("drawGradientString", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				LinearGradient gradient = new LinearGradient(0, 0, 99, 0,
						new int[] { 0xffff0000, 0xffffff00, 0xffffffff });
				g.setClip(x, y, w, h);
				g.setColor(Colors.WHITE);
				VectorGraphicsPainter.drawGradientString(g, "ABC", getFont(), h, getTranslatedMatrix(x, y), gradient,
						0xff, BlendMode.SRC_OVER, 0);
			}
		}, 50, 20);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawImage.
	 */
	@Test
	public static void testVGPainterDrawImageTranslatedGC() {
		testDrawer("drawImage", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.setClip(x, y, w, h);
				VectorImage image = getImage("test_buffered_vector_image_draw");
				Matrix matrix = new Matrix();
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				g.translate(x, y);
				VectorGraphicsPainter.drawImage(g, image, matrix);
				g.translate(-x, -y);
			}
		}, 50, 50, 0.01f);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawImage.
	 */
	@Test
	public static void testVGPainterDrawImageMatrix() {
		testDrawer("drawImage", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.setClip(x, y, w, h);
				VectorImage image = getImage("test_buffered_vector_image_draw");
				Matrix matrix = getTranslatedMatrix(x, y);
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				VectorGraphicsPainter.drawImage(g, image, matrix);
			}
		}, 50, 50, 0.01f);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawImage with several gradients.
	 */
	@Test
	public static void testVGPainterDrawGradientImageTranslatedGC() {
		testDrawer("drawGradientImage", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.setClip(x, y, w, h);
				VectorImage image = getImage("test_buffered_vector_image_draw_gradient");
				Matrix matrix = new Matrix();
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				g.translate(x, y);
				VectorGraphicsPainter.drawImage(g, image, matrix);
				g.translate(-x, -y);
			}
		}, 50, 50, 0.01f);
	}

	/**
	 * Tests some {@link VectorGraphicsPainter} algorithms: drawImage with several gradients.
	 */
	@Test
	public static void testVGPainterDrawGradientImageMatrix() {
		testDrawer("drawGradientImage", new Drawer() {
			@Override
			public void draw(GraphicsContext g, int x, int y, int w, int h) {
				g.setClip(x, y, w, h);
				VectorImage image = getImage("test_buffered_vector_image_draw_gradient");
				Matrix matrix = getTranslatedMatrix(x, y);
				matrix.preScale(w / image.getWidth(), h / image.getHeight());
				VectorGraphicsPainter.drawImage(g, image, matrix);
			}
		}, 50, 50, 0.01f);
	}

	private static void testDrawer(String label, Drawer drawer, int w, int h) {
		testDrawer(label, drawer, w, h, 0);
	}

	private static void testDrawer(String label, Drawer drawer, int w, int h, float tolerance) {
		try (BufferedVectorImage image = new BufferedVectorImage(w + 2, h + 3)) {

			Display display = Display.getDisplay();
			GraphicsContext g = display.getGraphicsContext();
			GraphicsContext ig = image.getGraphicsContext();

			// draw expected area on the left and virtual area on the right
			int padding = h;
			int x = g.getWidth() / 2 - w - padding;
			int y = g.getHeight() / 2;
			drawer.draw(g, x, y, w, h);
			drawer.draw(ig, 1, 2, w, h); // (1,2) to test the translation

			// draw virtual area on destination
			g.reset();
			Matrix m = new Matrix();
			m.setTranslate(x + w + padding - 1, y - 2);
			VectorGraphicsPainter.drawImage(g, image, m);

			// visualize the result
			display.flush();

			// compare areas
			TestUtilities.compareAreas(label, x, y, x + w + padding, y, w, h, tolerance);
		}
	}

	private static Matrix getTranslatedMatrix(int x, int y) {
		Matrix matrix = new Matrix();
		matrix.setTranslate(x, y);
		return matrix;
	}

	private static Path getPath(int w, int h) {
		Path path = new Path();
		path.lineTo(w, 0);
		path.lineTo(w, h);
		path.lineTo(0, h);
		path.close();
		return path;
	}

	private static VectorFont getFont() {
		return VectorFont.loadFont("/fonts/firstfont.ttf");
	}

	private static VectorImage getImage(String name) {
		return VectorImage.getImage("/com/microej/microvg/test/" + name + ".xml");
	}
}
