/*
 * Java
 *
 * Copyright 2023 MicroEJ Corp. All rights reserved.
 * Use of this source code is governed by a BSD-style license that can be found with this software.
 */
package ej.event;

import java.lang.Thread.UncaughtExceptionHandler;

/**
 * EventQueue is an asynchronous communication interface between the native world and the Java world based on events. It
 * allows users to send events from the native world to the Java world.
 * <p>
 * A FIFO mechanism is implemented on the native side and is system specific. The user can offer events to this FIFO.
 * <p>
 * Event notifications are handled using event listeners (Observer design pattern). The application code has to register
 * {@link EventQueueListener} to be notified when new events are coming. To do so, the user can call
 * {@link EventQueue#registerListener(EventQueueListener, int)}.
 * <p>
 * Then the pump automatically retrieves new events pushed in the FIFO and notifies the application via the
 * {@link EventQueueListener}.
 * <p>
 * EventQueue runs on a dedicated Java thread to forward and process events. Application event listener's calls are done
 * in he context of this thread. This thread is started when method {@link EventQueue#getInstance()} is called.
 * <p>
 * Events reading operations are done using the SNI mechanism. The thread is suspended if the events FIFO is empty. The
 * thread resume is done by the native part when a new event is sent if the events FIFO was previously empty.
 * <p>
 * <b> Event format </b>
 * <p>
 * An event is composed of a type and some data. The type identifies the listener that will handle the event. The data
 * is application specific and passed to the listener. The items stored in the FIFO buffer are integers (4 bytes). There
 * are two kinds of events that can be sent over the Event Queue:
 * <ul>
 * <li>Standard event: an event with data that fits on 24 bits. The event is stored in the FIFO as a single 4 bytes
 * item.</li>
 * <li>Extended event: an event with data that does not fit on 24 bits. The event is stored in the FIFO as multiple 4
 * bytes items.</li>
 * </ul>
 *
 * <pre>
 * +--------------+----------+---------------------------------------------------+
 * | Extended (1) | Type (7) | Data (if Extended==0), Length (if Extended==1) (24) |
 * +--------------+----------+---------------------------------------------------+
 * ...
 * +-----------------------------------------------------------------------------+
 * |                  Extended Data for extended events (32)                     | (x integers)
 * +-----------------------------------------------------------------------------+
 * </pre>
 *
 * Format explanation:
 *
 * <ul>
 * <li>Extended (1 bit): event kind flag (0 for non-extended event, 1 for extended event).</li>
 * <li>Type (7 bits): event type, which allows to find the corresponding event listener.</li>
 * <li>Length (24 bits): length of the data in bytes (for extended events only).</li>
 * <li>Data (24 bits): standard event data (for standard events only).</li>
 * <li>Extended data (`Length` bytes): extended event data (for extended events only).</li>
 * </ul>
 *
 * <p>
 *
 * For more information, please have a look at our
 * <a href="https://docs.microej.com/en/latest/VEEPortingGuide/eventQueue.html">Event Queue documentation</a>.
 *
 */
public class EventQueue {

	/**
	 * Creates a pump.
	 */
	private EventQueue() {

	}

	/**
	 * Sets a handler for the exceptions that occur during event reading or execution.
	 *
	 * @param uncaughtExceptionHandler
	 *            the uncaught exception handler to set.
	 */
	public void setUncaughtExceptionHandler(UncaughtExceptionHandler uncaughtExceptionHandler) {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

	/**
	 * Sets the default {@code listener}. It will receive all the events whose type is not handled by any registered
	 * listener.
	 *
	 * @param listener
	 *            the listener to set.
	 */
	public void setDefaultListener(EventQueueListener listener) {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

	/**
	 * Registers a {@code listener} that will received the events of a {@code type}.
	 * <p>
	 * The same listener can be registered for several types.
	 *
	 * @param listener
	 *            the listener to register.
	 * @param type
	 *            the type handled by the {@code listener}.
	 * @throws IllegalArgumentException
	 *             if the given {@code type} is lower than <code>0</code> or higher than <code>127</code>.
	 * @throws IllegalArgumentException
	 *             if there is already a listener for the event {@code type}.
	 */
	public synchronized void registerListener(EventQueueListener listener, int type) {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

	/**
	 * Inserts an event to the FIFO.
	 * <p>
	 * The {@code data} must not exceed 24 bits. Otherwise use {@link #offerExtendedEvent(int, byte[])}.
	 *
	 * @param type
	 *            the type of the event.
	 * @param data
	 *            the data of the event.
	 * @throws IllegalStateException
	 *             if the FIFO is full.
	 * @throws IllegalArgumentException
	 *             if the given {@code type} is lower than <code>0</code>.
	 * @throws IllegalArgumentException
	 *             if the {@code data} exceeds 24 bits.
	 */
	public void offerEvent(int type, int data) {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

	/**
	 * Inserts an extended event to the FIFO.
	 * <p>
	 * When the data exceeds 24 bits, it can be split into several integers. These integers are pushed consecutively to
	 * the FIFO. The listener will receive all these integers at once and will be able to rebuild a complex data.
	 *
	 * @param type
	 *            the type of the events.
	 * @param data
	 *            the data of the events.
	 * @throws IllegalStateException
	 *             if the FIFO is full.
	 * @throws IllegalArgumentException
	 *             if the given {@code type} is lower than <code>0</code>.
	 * @throws IllegalArgumentException
	 *             if the length of {@code data} buffer exceeds 24 bits.
	 */
	public void offerExtendedEvent(int type, byte[] data) {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

	/**
	 * Gets an instance of a started EventQueue.
	 *
	 * @return the instance of the queue.
	 */
	public static synchronized EventQueue getInstance() {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

	/**
	 * Gets an unique Id to register a Listener.
	 *
	 * @return the unique Id corresponding to the name.
	 * @throws IllegalStateException
	 *             if the maximum number of types (<code>127</code>) has been reached.
	 */
	public synchronized int getNewType() {
		throw new RuntimeException(); // NOSONAR method will be overridden by implementation.
	}

}
